#!/bin/bash
# detect-sev-snp.sh - Check if AMD SEV-SNP is available on this system
#
# Exit codes:
#   0 - SEV-SNP is available
#   1 - SEV-SNP is NOT available

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo "=== AMD SEV-SNP Detection ==="
echo ""

CHECKS_PASSED=0
CHECKS_TOTAL=2

# Check 1: /dev/sev-guest device
echo -n "Checking /dev/sev-guest device... "
if [[ -c /dev/sev-guest ]]; then
    echo -e "${GREEN}FOUND${NC}"
    ((CHECKS_PASSED++))
else
    echo -e "${RED}NOT FOUND${NC}"
    echo "  └─ The /dev/sev-guest device is required for SEV-SNP attestation"
    echo "     This device is created by the sev-guest kernel module"
fi

# Check 2: Kernel module
echo -n "Checking kernel modules (sev-guest or ccp)... "
if lsmod 2>/dev/null | grep -qE '^(sev_guest|ccp)\s'; then
    LOADED_MODULE=$(lsmod | grep -oE '^(sev_guest|ccp)' | head -1)
    echo -e "${GREEN}LOADED${NC} ($LOADED_MODULE)"
    ((CHECKS_PASSED++))
elif [[ -d /sys/module/sev_guest ]] || [[ -d /sys/module/ccp ]]; then
    echo -e "${GREEN}BUILT-IN${NC}"
    ((CHECKS_PASSED++))
else
    echo -e "${RED}NOT LOADED${NC}"
    echo "  └─ Try: modprobe sev-guest"
fi

# Summary
echo ""
echo "=== Summary ==="
if [[ $CHECKS_PASSED -eq $CHECKS_TOTAL ]]; then
    echo -e "${GREEN}SEV-SNP is AVAILABLE${NC} ($CHECKS_PASSED/$CHECKS_TOTAL checks passed)"
    echo ""
    echo "You can now generate an attestation report:"
    echo "  ./scripts/generate-report.sh ./attestation-output"
    exit 0
else
    echo -e "${RED}SEV-SNP is NOT AVAILABLE${NC} ($CHECKS_PASSED/$CHECKS_TOTAL checks passed)"
    echo ""
    echo "Possible reasons:"
    echo "  - This VM is not running with SEV-SNP protection"
    echo "  - SEV-SNP is not enabled in the hypervisor/BIOS"
    echo "  - The kernel does not have SEV-SNP guest support"
    echo "  - The CPU does not support SEV-SNP (requires AMD EPYC 7003+ series)"
    exit 1
fi
