# SEV-SNP Attestation Error Codes and Troubleshooting

This guide covers common errors encountered during SEV-SNP attestation and their solutions.

## Detection Errors

### /dev/sev-guest not found

**Symptoms:**
```
Error: /dev/sev-guest not found
SEV-SNP is not available on this system.
```

**Causes:**
1. VM is not running with SEV-SNP protection
2. Kernel module not loaded
3. Missing kernel support

**Solutions:**

1. **Check if VM has SEV-SNP enabled:**
   ```bash
   dmesg | grep -i sev
   # Should show: "SEV-SNP: SEV-SNP detected" or similar
   ```

2. **Load the kernel module:**
   ```bash
   sudo modprobe sev-guest
   ```

3. **Verify kernel config:**
   ```bash
   grep CONFIG_SEV_GUEST /boot/config-$(uname -r)
   # Should show: CONFIG_SEV_GUEST=y or CONFIG_SEV_GUEST=m
   ```

4. **Check with hypervisor admin** that SEV-SNP is enabled for your VM.

### Kernel module not loaded

**Symptoms:**
```
Checking kernel modules (sev-guest or ccp)... NOT LOADED
```

**Solutions:**

1. **Load the module:**
   ```bash
   sudo modprobe sev-guest
   # or for older kernels:
   sudo modprobe ccp
   ```

2. **Make it persistent:**
   ```bash
   echo "sev-guest" | sudo tee /etc/modules-load.d/sev-guest.conf
   ```

### SEV-SNP not detected in firmware/CPU

**Symptoms:**
```
Checking SEV-SNP firmware/CPU support... NOT DETECTED
```

**Causes:**
1. Running on non-AMD or older AMD CPU
2. SEV-SNP disabled in BIOS
3. Hypervisor doesn't support SEV-SNP

**Solutions:**

1. **Verify CPU support:**
   ```bash
   # SEV-SNP requires AMD EPYC 7003 (Milan) or newer
   cat /proc/cpuinfo | grep "model name" | head -1
   ```

2. **Contact system administrator** to enable SEV-SNP in BIOS/hypervisor.

## Report Generation Errors

### Permission denied accessing /dev/sev-guest

**Symptoms:**
```
Error: Could not open /dev/sev-guest: Permission denied
```

**Solutions:**

1. **Run as root:**
   ```bash
   sudo ./scripts/generate-report.sh ./output
   ```

2. **Add user to sev group:**
   ```bash
   sudo usermod -a -G sev $USER
   # Log out and back in
   ```

3. **Check device permissions:**
   ```bash
   ls -la /dev/sev-guest
   # Fix if needed:
   sudo chmod 660 /dev/sev-guest
   sudo chown root:sev /dev/sev-guest
   ```

### snpguest not found

**Symptoms:**
```
Error: snpguest not found
```

**Solutions:**

1. **Install snpguest:**
   ```bash
   cargo install snpguest
   ```

2. **Add cargo bin to PATH:**
   ```bash
   export PATH="$HOME/.cargo/bin:$PATH"
   ```

3. **Install Rust if needed:**
   ```bash
   curl --proto '=https' --tlsv1.2 -sSf https://sh.rustup.rs | sh
   ```

### IOCTL failure

**Symptoms:**
```
Error: IOCTL SNP_GET_REPORT failed
```

**Causes:**
1. Incorrect kernel/firmware version
2. SEV-SNP not properly initialized

**Solutions:**

1. **Check dmesg for errors:**
   ```bash
   dmesg | grep -i "sev\|snp" | tail -20
   ```

2. **Verify firmware version:**
   ```bash
   # On host (requires access)
   cat /sys/kernel/security/sev/api_major
   cat /sys/kernel/security/sev/api_minor
   ```

## Certificate Fetch Errors

### Failed to fetch certificates from AMD KDS

**Symptoms:**
```
Error: Failed to fetch certificate chain
curl: (7) Failed to connect to kdsintf.amd.com
```

**Causes:**
1. Network connectivity issues
2. Firewall blocking HTTPS
3. AMD KDS service unavailable

**Solutions:**

1. **Test connectivity:**
   ```bash
   curl -I https://kdsintf.amd.com
   ```

2. **Check firewall rules** for outbound HTTPS (port 443).

3. **Use a proxy if required:**
   ```bash
   export https_proxy=http://proxy.example.com:8080
   ```

### VCEK not found for this chip

**Symptoms:**
```
Error: Failed to fetch VCEK certificate
HTTP 404 Not Found
```

**Causes:**
1. CHIP_ID not registered with AMD
2. Invalid TCB version in request
3. Report from development/test hardware

**Solutions:**

1. **Verify CHIP_ID extraction:**
   ```bash
   xxd -p -s 416 -l 64 report.bin | tr -d '\n'
   ```

2. **Contact AMD** if using production hardware without VCEK.

## Chain Verification Errors

### ARK self-signature failed

**Symptoms:**
```
✗ ARK self-signature verification FAILED
```

**Causes:**
1. Corrupted ARK certificate
2. Wrong certificate file

**Solutions:**

1. **Re-fetch certificates:**
   ```bash
   rm -rf ./output/certs
   ./scripts/fetch-certificates.sh ./output/report.bin ./output
   ```

2. **Verify certificate format:**
   ```bash
   openssl x509 -in certs/ark.pem -text -noout
   ```

### VCEK not signed by ASK

**Symptoms:**
```
✗ VCEK signature verification FAILED
```

**Causes:**
1. VCEK from different platform than ASK
2. Mixed Milan/Genoa certificates

**Solutions:**

1. **Ensure consistent platform:**
   - Milan requires Milan certificates
   - Genoa requires Genoa certificates

2. **Re-fetch all certificates** from scratch.

## Report Verification Errors

### Report signature verification failed

**Symptoms:**
```
✗ Report signature verification FAILED
Verified NOK
```

**Causes:**
1. Report tampered or corrupted
2. Wrong VCEK certificate
3. Certificate/report mismatch

**Solutions:**

1. **Generate fresh report and certificates:**
   ```bash
   ./scripts/full-attestation.sh ./new-output
   ```

2. **Verify VCEK matches chip:**
   ```bash
   # Compare CHIP_ID in report with VCEK
   xxd -p -s 416 -l 64 report.bin | tr -d '\n'
   openssl x509 -in certs/vcek.pem -text | grep -A1 hwID
   ```

### TCB mismatch

**Symptoms:**
```
Warning: TCB version mismatch between report and VCEK
```

**Causes:**
1. Firmware updated since VCEK was issued
2. TCB downgraded (security concern)

**Solutions:**

1. **Fetch fresh VCEK** with current TCB values.

2. **If TCB downgraded**, investigate why firmware was rolled back.

## General Troubleshooting

### Enable verbose output

Set environment variable for more details:
```bash
export SEV_DEBUG=1
./scripts/full-attestation.sh ./output
```

### Collect diagnostic information

```bash
# System info
uname -a
cat /proc/cpuinfo | grep -E "vendor|model name|flags" | head -5

# SEV status
dmesg | grep -i sev
cat /sys/kernel/security/sev/* 2>/dev/null
ls -la /dev/sev*

# Kernel modules
lsmod | grep -E "sev|ccp"
```

### Getting help

1. Check [AMD SEV-SNP documentation](https://developer.amd.com/sev/)
2. Review [snpguest issues](https://github.com/virtee/snpguest/issues)
3. Consult your cloud provider's confidential computing documentation
