# SEO Optimizer Pro

**AI-powered SEO content optimization for both Google ranking and AI search (AEO).**

Analyze and optimize content with AI for search visibility across traditional Google results and emerging AI search platforms (ChatGPT, Google AI Overviews, Claude, etc).

---

## 🚀 Why This Skill?

### Problem Statement
SEO optimization requires balancing multiple objectives:
- Content must rank on Google with proper technical SEO
- AI search optimization (AEO) is a new frontier—getting cited in ChatGPT, Google AI Overviews, Claude responses
- Manual analysis of readability, keyword density, heading structure is time-consuming
- Unclear what will actually impact rankings vs. what's just best practice
- Need both technical optimization AND compelling AI-optimized content

### The Solution
Analyze and optimize content using AI combined with technical SEO audits. Get specific recommendations for both traditional Google ranking AND emerging AI search visibility.

### Why This Matters
- **AI Search is Growing**: ChatGPT, Google AI Overviews, Claude are becoming major discovery channels
- **2026 Trend**: Quality content with proper structure ranks better in both systems
- **Data-Driven**: Real recommendations based on actual SEO metrics, not just generic advice
- **Dual Optimization**: One analysis optimizes for both Google AND AI search platforms

---

## ✨ What You Get

### Comprehensive SEO Analysis
Every analysis includes:
- 📊 **Readability Score** - Flesch-Kincaid grade level and readability metrics
- 🔑 **Keyword Analysis** - Density tracking and natural integration suggestions
- 📐 **Content Structure** - Heading hierarchy, paragraph length, word count
- 🎯 **Technical SEO** - Meta tags, link analysis, structured data
- 🤖 **AI Search Optimization** - Recommendations for ChatGPT citations, AI Overviews

### Model-Agnostic AI Integration
Pass any model ID — provider is auto-detected from the model name prefix.
No hardcoded list; new models work without updates.

| Provider | Tested Models | Prefix |
|----------|--------------|--------|
| **Anthropic** | claude-opus-4-6, claude-sonnet-4-6, claude-haiku-4-5-20251001 | `claude-` |
| **OpenAI** | gpt-5.2-pro, gpt-5.2, gpt-5.1 | `gpt-`, `o1`, `o3` |
| **Google** | gemini-2.5-pro, gemini-2.5-flash | `gemini-` |
| **Mistral** | mistral-large-latest, mistral-small-latest | `mistral-`, `mixtral-` |
| **DeepSeek** | deepseek-chat, deepseek-reasoner | `deepseek-` |
| **xAI** | grok-4-1-fast, grok-3-beta | `grok-` |
| **MiniMax** | MiniMax-M2.1 | `MiniMax`, `minimax` |
| **Qwen** | qwen3.5-plus, qwen3-max-instruct | `qwen` |
| **Meta Llama** | meta-llama/llama-4-maverick, meta-llama/llama-3.3-70b-instruct | `meta-llama/`, `llama-` |

Features:
- **Title Optimization** - SEO-friendly titles (40-60 chars)
- **Meta Descriptions** - Auto-generated descriptions (150-160 chars)
- **H1 Suggestions** - Optimized heading tags
- **Key Points Expansion** - What topics to expand for better coverage
- **Keyword Integration** - Natural ways to incorporate target keywords
- **Internal Linking Ideas** - Strategic internal linking opportunities

### Answer Engine Optimization (AEO)
Five specific recommendations to:
- Get your content featured in AI search results
- Appear as cited sources in ChatGPT, Google AI Overviews, Claude
- Optimize for question-answer format (how AI searches work)
- Build authority signals that AI systems recognize
- Structure content for AI comprehension and citation

### Technical Metrics
Analyze:
- Word count and content length
- Sentence and paragraph structure
- Keyword density by target keyword
- Heading structure (H1, H2, H3...)
- Readability (Flesch-Kincaid grade)
- Link analysis (internal/external)
- Content organization

---

## 📊 Real-World Example

```
ANALYZING: Blog post about "Cloud Storage Solutions"

═══════════════════════════════════════════════════════════════

📊 METRICS
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
• Word Count: 847 words
• Readability Score: 68/100 (Good)
• Flesch-Kincaid Grade: 8.2 (High school level - good)
• Avg Paragraph: 32 words (Optimal range)

🔑 KEYWORD DENSITY
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
• cloud storage: 1.8% ✅ (Optimal)
• data security: 0.8% (Increase to 1-2%)
• backup solutions: 0.4% (Too low)

💡 TOP SUGGESTIONS
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🟡 [KEYWORDS] "backup solutions" appears only 0.4% of the time
   Current: 0.4% → Recommended: 1-2%
   Impact: Better keyword relevance signal

🟢 [TECHNICAL] Heading structure is well organized
   Current: H1 (1), H2 (4), H3 (2) ✅
   Impact: Excellent for SEO and accessibility

🟡 [CONTENT] Consider expanding "data encryption" section
   Current: 1 paragraph → Recommended: 2-3 paragraphs
   Impact: More thorough topic coverage, higher authority

🤖 AI SEARCH OPTIMIZATION (AEO)
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

1. Add a FAQ section with common questions about cloud storage
   → This format gets cited in ChatGPT and Google AI Overviews

2. Include specific comparison tables (Google vs AWS vs Azure)
   → Structured data that AI systems cite as authoritative

3. Add a "Quick Summary" box with key takeaways
   → AI search pulls these summaries for AI-generated responses

4. Link to authoritative sources in your references
   → Shows expertise, increases citation likelihood

5. Structure content as problem → solution → benefits
   → Matches how AI systems generate responses

🎯 ESTIMATED IMPACT
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
Medium Impact - Implementing 3-5 suggestions could improve ranking
potential by 10-20%. Key factors: Expanding keyword coverage,
enhancing structure for AI readability, adding comparison tables.
```

---

## Use Cases

### Content Marketing
- Optimize blog posts for both Google and AI search
- Ensure content gets cited in ChatGPT, Google AI Overviews
- A/B test content structure recommendations
- Scale content optimization across multiple writers

### SEO Professionals
- Audit client websites for optimization opportunities
- Provide data-driven recommendations with impact estimates
- Build optimization reports with specific, actionable steps
- Track progress across multiple keyword targets

### Product Documentation
- Optimize docs for both search and AI assistant discovery
- Ensure technical content appears in AI search results
- Improve readability scores for accessibility
- Structure docs for AI system comprehension

### E-Commerce
- Optimize product descriptions for Google Shopping and AI search
- Improve category page rankings with strategic content
- Enhance FAQ sections for AEO visibility
- Increase organic traffic from emerging AI search channels

### News & Publishing
- Optimize articles for news search and AI citations
- Improve readability for different audience levels
- Structure content for AI system extraction and citation
- Track SEO performance metrics over time

---

## 🔐 Credentials Required

You need **one API key** — for whichever provider/model you want to use:

| Provider | Env Variable | Prefix | Get Key |
|----------|-------------|--------|---------|
| Anthropic | `ANTHROPIC_API_KEY` | `claude-` | https://console.anthropic.com |
| OpenAI | `OPENAI_API_KEY` | `gpt-`, `o1`, `o3` | https://platform.openai.com |
| Google | `GOOGLE_API_KEY` | `gemini-` | https://aistudio.google.dev |
| Mistral | `MISTRAL_API_KEY` | `mistral-`, `mixtral-` | https://console.mistral.ai |
| DeepSeek | `DEEPSEEK_API_KEY` | `deepseek-` | https://platform.deepseek.com |
| xAI | `XAI_API_KEY` | `grok-` | https://console.x.ai |
| MiniMax | `MINIMAX_API_KEY` | `minimax`, `MiniMax` | https://platform.minimax.io |
| Qwen/Alibaba | `DASHSCOPE_API_KEY` | `qwen` | https://dashscope.aliyun.com |
| OpenRouter | `OPENROUTER_API_KEY` | `meta-llama/`, `llama-` | https://openrouter.ai |

**Only set the key for the provider you're using.** For example, to use Claude:
```bash
export ANTHROPIC_API_KEY=sk-ant-...
```

Or to use GPT:
```bash
export OPENAI_API_KEY=sk-...
```

The skill itself is free. You pay the provider directly for API usage at their standard rates.

### ⚠️ Important Privacy Notice

**Your content IS sent to third-party AI providers.** Before using this skill with sensitive content, review the privacy policy of your chosen provider:

- **Anthropic Privacy**: https://www.anthropic.com/legal/privacy
- **OpenAI Privacy**: https://openai.com/policies/privacy-policy
- **Google Gemini Terms**: https://ai.google.dev/gemini-api/terms
- **OpenRouter Privacy**: https://openrouter.ai/privacy
- **Mistral Terms**: https://mistral.ai/terms/

**Recommendation**: Test with non-sensitive sample content first. Each provider has different data retention policies.

---

## 🚀 Quick Start

### 1. Access the Skill
Available for free on ClawhHub - no subscription required

### 2. Set Up

**Required:**
- API key for your chosen provider (see Credentials table above)

**Optional:**
- Content to analyze (HTML or plain text)
- Target keywords (recommended for better analysis)

### 3. Python Usage

```python
from skills.seo_optimizer import SEOOptimizer

# Initialize with default Claude model
optimizer = SEOOptimizer()

# Or use a different model
optimizer = SEOOptimizer(model="gpt-5.2-pro")  # Use GPT-5.2
# optimizer = SEOOptimizer(model="gemini-3-pro")  # Use Gemini 3
# optimizer = SEOOptimizer(model="mistral-large-2501")  # Use Mistral

# Analyze content
results = optimizer.analyze_content(
    content="""
    <h1>Cloud Storage Solutions</h1>
    <p>Cloud storage has revolutionized how we store and access data...</p>
    ...(your content)...
    """,
    url="https://example.com/cloud-storage",
    target_keywords=["cloud storage", "data security", "backup solutions"]
)

# Get formatted results
print(optimizer.format_results(results))

# Access detailed metrics
print(f"Readability: {results.metrics.readability_score}/100")
print(f"Word Count: {results.metrics.word_count}")

# Get AI suggestions
print(results.content_optimization)

# Get AEO recommendations
for i, rec in enumerate(results.aeo_recommendations, 1):
    print(f"{i}. {rec}")
```

**Tested Models (any model matching a provider prefix also works):**
- `claude-opus-4-6` (Recommended — best quality)
- `claude-sonnet-4-6`
- `claude-haiku-4-5-20251001` (Default — fastest/cheapest)
- `gpt-5.2-pro`, `gpt-5.2`, `gpt-5.1`
- `gemini-2.5-pro`, `gemini-2.5-flash`
- `mistral-large-latest`, `mistral-small-latest`
- `deepseek-chat`, `deepseek-reasoner`
- `grok-4-1-fast`, `grok-3-beta`
- `MiniMax-M2.1`
- `qwen3.5-plus`, `qwen3-max-instruct`
- `meta-llama/llama-4-maverick`, `meta-llama/llama-3.3-70b-instruct`

### 4. CLI Usage

```bash
# Analyze a file
seo-optimizer analyze "blog-post.html" --keywords "cloud storage" "data security"

# Generate optimization report
seo-optimizer analyze "article.txt" --report --output report.json

# Quick AEO analysis
seo-optimizer aeo "content.html" --keywords "topic-1" "topic-2"
```

---

## 🤖 AI Search Optimization (AEO) Deep Dive

### What is AEO?
Answer Engine Optimization is the practice of optimizing content to appear in AI-generated search results and as cited sources in AI chatbots (ChatGPT, Google AI Overviews, Claude, Perplexity, etc).

### Why AEO Matters
- **ChatGPT** can cite your content in responses (if properly formatted)
- **Google AI Overviews** pull information from highly-relevant, authoritative sources
- **Claude** and other AI systems prioritize structured, clear information
- **2026 Reality**: AI search is becoming a primary discovery channel alongside Google

### AEO Best Practices (From Analysis)

**1. Structure for AI Comprehension**
- Use clear heading hierarchy (H1 → H2 → H3)
- Break content into logical sections
- Use lists and bullet points liberally
- AI systems parse structured content better

**2. Direct Answers Format**
- Start with direct answer to the main question
- Support with detailed explanation
- Include supporting evidence/sources
- Match how humans ask questions

**3. Structured Data**
- FAQ schema for Q&A format
- Product schema for product pages
- Organization schema for company info
- Schema.org markup helps AI systems understand context

**4. Authority Signals**
- Link to authoritative sources
- Cite peer-reviewed research
- Include author credentials
- AI systems recognize and cite authoritative sources

**5. Comparison Tables**
- Structured comparisons get cited in AI responses
- Use tables for side-by-side information
- Include pros/cons lists
- AI systems extract and cite these directly

---

## 🔒 Security & Privacy

### Data Handling
- ✅ Content analyzed in-memory only
- ✅ Not stored or transmitted to external servers
- ✅ Claude API processes via encrypted HTTPS
- ✅ Analysis results stored locally only

### API Keys
- ✅ API keys stored in environment variables
- ✅ Never logged or exposed
- ✅ Each provider uses their official API
- ✅ Zero-knowledge architecture

### Compliance
- ✅ GDPR compliant (no data storage)
- ✅ CCPA compliant (no data collection)
- ✅ Works with proprietary content
- ✅ No third-party data sharing

---

## ❓ FAQ

**Q: How do you calculate readability score?**
A: We use the Flesch-Kincaid grade level formula and convert it to a 0-100 scale. Higher scores = easier to read.

**Q: Are these recommendations guaranteed to improve rankings?**
A: No ranking is guaranteed, but these suggestions are based on proven SEO factors. Impact varies by industry, competition, and current content quality.

**Q: What's the difference between SEO and AEO?**
A: SEO optimizes for Google's algorithm. AEO optimizes for AI chatbots that generate responses. They overlap significantly—good content for humans is usually good for both.

**Q: Can I use this for languages other than English?**
A: The readability analysis is English-optimized, but keyword analysis and AI suggestions work for any language.

**Q: How long should content be?**
A: It depends on topic complexity. Generally: 300+ words minimum, 1000+ for comprehensive coverage, 2000+ for pillar/authority pages.

**Q: What if I have a very long document?**
A: The skill analyzes the first 2000 characters by default. For longer documents, split into sections and analyze each separately.

**Q: Does this check technical SEO like page speed?**
A: This skill focuses on content optimization. For technical SEO (speed, mobile, crawlability), use tools like Google PageSpeed Insights.

**Q: Can I optimize for multiple keywords at once?**
A: Yes! Pass a list of target keywords. We'll analyze density for each and provide keyword-specific recommendations.

**Q: How accurate is the AEO scoring?**
A: AEO is still evolving. Our recommendations are based on 2026 best practices from AI search optimization research.

**Q: Does this skill generate SEO content from scratch?**
A: No, this skill analyzes and optimizes existing content. To generate new content, use other AI writing tools first.

---

## 🗺️ Roadmap

### ✅ Current Release (v1.0.2)
- Content readability analysis (Flesch-Kincaid)
- Keyword density tracking
- Technical SEO audit
- Multi-provider AI optimization (Claude, GPT, Gemini, Llama, Mistral)
- AEO-specific recommendations
- Impact estimation

### 🚧 Coming Soon
- **Competitor analysis** - Compare your content to top Google results
- **Bulk analysis** - Analyze multiple pages in one run
- **Export formats** - PDF reports, JSON data, CSV analysis
- **Multi-language readability** - Support beyond English

### 🔮 Future
- **Schema generation** - Auto-generate JSON-LD schema
- **Link analysis** - Deep internal/external link auditing
- **Content gap analysis** - Identify missing topics in your content

---

## 📚 Resources

### Support
- Email: support@unisai.vercel.app
- Website: https://unisai.vercel.app

---

## 📄 License & Terms

This skill is **free and open to all users** on ClawhHub.

### ✅ You CAN:
- Use the skill for free, unlimited times
- Analyze unlimited content
- Export optimization reports
- Use for commercial purposes
- Use for personal and business projects

### ❌ You CANNOT:
- Redistribute or resell the skill
- Modify the source code without permission
- Reverse engineer the skill

**Full Terms**: See [LICENSE.md](LICENSE.md)

---

## 🚀 Get Started

1. Access this free skill on ClawhHub (no subscription needed)
2. Install the SDK for your provider: `pip install anthropic` (or openai, google-generativeai, mistralai)
3. Set the API key for your chosen provider (see Credentials table above)
4. Analyze your first piece of content

---

## 🏷️ Tags

**Primary**: seo, content-optimization, aeo, ai-search, copywriting

**Categories**: seo-tools, content-analysis, marketing-tech, ai-optimization

**Features**: keyword-analysis, readability-scoring, meta-tag-generation, technical-seo, ai-search-optimization

---

## 📝 Changelog

### [1.0.7] - 2026-02-27

#### 🏗️ Model-Agnostic Architecture
- Provider auto-detected from model name prefix — no hardcoded whitelist
- Any new model works automatically without code changes
- Added DeepSeek, xAI Grok, MiniMax, Qwen as first-class providers
- Updated Claude to 4.6 series (claude-opus-4-6, claude-sonnet-4-6)
- Simplified internals — all OpenAI-compat providers share one code path

### [1.0.6] - 2026-02-14

#### 🔒 Security & Transparency Improvements
- **Source Code Included** - Fixed license contradiction; source code IS included for transparency and security audits
- **License Clarification** - Changed from "proprietary" to "free-to-use" with clear modification rights for personal use
- **Data Sharing Warnings** - Added explicit warnings about third-party AI provider data sharing with links to privacy policies
- **Metadata Consistency** - Fixed registry metadata to accurately reflect required environment variables
- **Install Recommendations** - Updated requirements.txt and docs to recommend selective SDK installation (not all packages)

#### 📝 Documentation Updates
- Added GitHub repository and verifiable support contacts
- Clarified which API key is required for which model
- Added provider privacy policy links for informed consent
- Updated all version references to 1.0.6

### [1.0.2] - 2026-02-14

#### ✨ Multi-Model Support
- **Claude 4.5 Series** - Opus, Sonnet, Haiku models
- **GPT-5.2 Series** - Pro, Thinking, Instant variants
- **Gemini 2.5/3.0** - Google's latest models
- **Llama 3.2/3.3** - Meta's open-source models
- **Mistral Large** - Mistral AI's flagship model
- **Model Selection** - Choose your preferred AI model for analysis

#### 🔧 Technical Improvements
- Model-agnostic architecture for flexibility
- Support for 12+ leading AI models
- Dynamic model validation
- Improved API compatibility

### [1.0.0] - 2026-02-02

#### ✨ Initial Release
- **Content Analysis** - Readability, word count, paragraph structure
- **Keyword Tracking** - Density analysis for target keywords
- **Technical SEO** - Heading structure, link analysis, meta tags
- **AI Integration** - AI-powered title, description, H1 suggestions
- **AEO Optimization** - 5 specific recommendations for AI search visibility
- **Impact Estimation** - Predicted ranking improvement from optimizations
- **Beautiful Reports** - Formatted analysis with clear recommendations

#### 🎨 2026 Update
- Current 2026 models and pricing
- Professional, technical tone
- Focus on AEO (Answer Engine Optimization)

#### 🔒 Security & IP Protection
- IP watermark: `PROPRIETARY_SKILL_SEO_OPTIMIZER_2026`
- Zero data storage
- HTTPS-only API communication
- Environment variable security

---

**Last Updated**: February 27, 2026
**Current Version**: 1.0.7
**Status**: Active & Maintained

© 2026 UnisAI. All rights reserved.
