/**
 * Send USD Skill - Transfer USD between agents
 * @module send-usd
 */

/**
 * Execute the USD transfer
 * @param {Object} ctx - Execution context
 * @param {Object} ctx.params - Input parameters
 * @param {string} ctx.params.from_agent - Sender agent identifier
 * @param {string} ctx.params.to_agent - Recipient agent identifier
 * @param {number} [ctx.params.amount=1.00] - Amount in USD to transfer
 * @param {string} [ctx.params.memo] - Optional transaction note
 * @returns {Promise<Object>} Transfer result
 */
export async function execute(ctx) {
  const { from_agent, to_agent, amount = 1.0, memo = "" } = ctx.params;

  // Validate inputs
  if (!from_agent || typeof from_agent !== "string") {
    return {
      success: false,
      transaction_id: null,
      message: "Invalid from_agent: must be a non-empty string",
      error_code: "INVALID_SENDER",
    };
  }

  if (!to_agent || typeof to_agent !== "string") {
    return {
      success: false,
      transaction_id: null,
      message: "Invalid to_agent: must be a non-empty string",
      error_code: "INVALID_RECIPIENT",
    };
  }

  if (typeof amount !== "number" || amount < 0.01) {
    return {
      success: false,
      transaction_id: null,
      message: "Invalid amount: must be at least $0.01",
      error_code: "INVALID_AMOUNT",
    };
  }

  if (from_agent === to_agent) {
    return {
      success: false,
      transaction_id: null,
      message: "Cannot transfer to the same agent",
      error_code: "INVALID_RECIPIENT",
    };
  }

  try {
    // Generate transaction ID
    const transaction_id = `txn_${Date.now()}_${Math.random().toString(36).substring(2, 9)}`;

    // TODO: Integrate with your payment provider here
    // Example: const result = await paymentAPI.transferUSD(from_agent, to_agent, amount);

    // Simulated successful transfer
    const result = {
      success: true,
      transaction_id,
      amount,
      from_agent,
      to_agent,
      memo,
      timestamp: new Date().toISOString(),
      message: `Successfully transferred $${amount.toFixed(2)} USD from ${from_agent} to ${to_agent}`,
    };

    return result;
  } catch (error) {
    return {
      success: false,
      transaction_id: null,
      message: error.message || "Transfer failed",
      error_code: "TRANSFER_FAILED",
    };
  }
}
