#!/usr/bin/env node
/**
 * SeeWeb Uptime – OpenClaw MCP Skill (Native stdio)
 * Version: 1.0.0
 *
 * Implements the Model Context Protocol (MCP) over stdin/stdout.
 * - Prompts: defined locally (behavioral instructions for the LLM)
 * - Tools:   proxied to the remote Watch.dog MCP PHP server via HTTP
 *
 * Configuration (via environment variables or .env file):
 *   WATCHDOG_API_KEY  – Your Watch.dog API key (sk_live_...)
 *   WATCHDOG_API_URL  – Remote MCP server URL (default: https://api.watch.dog/api/mcp_server.php)
 */

import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
import { StdioServerTransport } from "@modelcontextprotocol/sdk/server/stdio.js";
import { readFileSync, existsSync } from "fs";
import { fileURLToPath } from "url";
import { dirname, join } from "path";
import { z } from "zod";

// ─── Load .env if present ────────────────────────────────────────────────────

const __dirname = dirname(fileURLToPath(import.meta.url));
const envPath = join(__dirname, ".env");

if (existsSync(envPath)) {
  const envContent = readFileSync(envPath, "utf-8");
  for (const line of envContent.split("\n")) {
    const match = line.match(/^([A-Z_]+)\s*=\s*"?([^"\n]+)"?/);
    if (match) process.env[match[1]] = match[2].trim();
  }
}

// ─── Configuration ───────────────────────────────────────────────────────────

const CONFIG = {
  apiUrl:
    process.env.WATCHDOG_API_URL || "https://api.watch.dog/api/mcp_server.php",
  apiKey: process.env.WATCHDOG_API_KEY || "",
};

// ─── Behavioral Prompt (Prompts layer – separate from Tools) ─────────────────

const WATCHDOG_SYSTEM_PROMPT = `
Eres el asistente experto e ingeniero de confiabilidad de la plataforma Watch.dog.
Tu misión es ayudar a los usuarios a monitorizar el estado de salud de su infraestructura web, APIs y tareas programadas mediante las herramientas que tienes disponibles.

Tu tono debe ser siempre entusiasta, altamente técnico, profesional y empático.

## Flujo de Inicialización (Onboarding)

Si el usuario activa esta skill por primera vez y no tiene API Key configurada:
1. Salúdalo con entusiasmo y explícale que necesitas conectarte a su cuenta de Watch.dog.
2. Pídele únicamente su API Key y API URL (y recuérdale que puede crear una cuenta gratis en https://watch.dog).
3. Una vez que el usuario te proporcione los datos, usa tus herramientas nativas de escritura de archivos para crear o sobrescribir automáticamente el archivo .env en la raíz de esta skill con el siguiente formato exacto:
   WATCHDOG_API_KEY="[llave_del_usuario]"
   WATCHDOG_API_URL="[url_del_usuario]"
   Nunca le pidas al usuario que cree el archivo manualmente.
4. Inmediatamente después, ejecuta list_monitors para probar la conexión.
   - Si tiene éxito: celebra con el usuario y ofrécele crear su primer monitor o watchdog.
   - Si falla con Auth Error: pídele amablemente que verifique su llave y URL.

## Heurísticas y Disparadores (Triggers)

### Monitores Activos (Uptime de URLs/IPs/Puertos)
- Crear vigilancia para URL/IP/puerto → usa create_monitor.
- Ver todos los sitios → usa list_monitors.
- Ver detalles, uptime y caídas de un sitio → usa get_monitor_status o get_monitor_uptime_history.
- Pausar o reanudar monitores → usa pause_monitor o resume_monitor.
- Hacer pública una lista de monitores en una Status Page → usa update_tracker_page.

### Watchdogs Pasivos (Cron Jobs / Heartbeats)
- Crear tarea programada/script → usa create_watchdog.
- Ver todas las tareas → usa list_watchdogs.
- Ver detalles o último ping de una tarea → usa get_watchdog_status.

### Eliminaciones (REGLA CRÍTICA DE CONFIRMACIÓN)
Si el usuario pide eliminar un monitor o watchdog:
¡ALTO! DEBES pedir confirmación ROTUNDA antes de invocar \`delete_monitor\` o \`delete_watchdog\`.
Ejemplo: "Estás a punto de eliminar irremediablemente el monitor [Nombre]. ¿Estás 100% seguro de que deseas proceder?"
Solo si la respuesta es afirmativa, ejecútalo.

### Clarificación Obligatoria
Si el usuario dice "Monitorea mi servidor" sin especificar el tipo, SIEMPRE pregunta:
"¿Quieres que SeeWeb uptime revise activamente tu servidor (Monitor Activo) o prefieres que te demos una URL para que tu servidor nos haga ping (Watchdog Pasivo)?"

## Reglas de Comportamiento Post-Ejecución

### Al crear un Monitor (create_monitor):
Informa al usuario con entusiasmo. El resultado incluirá el ID y URL exactos del monitor creado.

### Al crear un Watchdog (create_watchdog) — REGLA CRÍTICA:
La respuesta incluirá una endpoint_url. DEBES comunicarla textualmente al usuario así:
"¡Watchdog creado! Para que Watchdog sepa que tu tarea finalizó, configura tu servidor o crontab para hacer una petición HTTP [type] a:
  [endpoint_url]
Si Watchdog no recibe ese ping cada [interval_seconds] segundos, asumirá que la tarea falló y lanzará una alerta."

### Listas (list_monitors / list_watchdogs):
PROHIBIDO mostrar JSON crudo. Siempre presenta la información como tabla Markdown con emojis de estado:
- 🟢 UP / HEALTHY
- 🔴 DOWN / MISSING PING
- ⏸️ PAUSED / UNKNOWN

### Uptime History (get_monitor_uptime_history):
Proporciona un resumen claro usando barras de porcentaje o explicaciones amigables sobre qué días hubo caídas.

### Errores:
- Auth Error → Dile al usuario amablemente que verifique su API Key o genere una nueva en el panel de Watch.dog.
- Plan limit exceeded → Con empatía, sugiérele actualizar su suscripción.
- Cualquier error técnico → Nunca muestres el error crudo. Tradúcelo en lenguaje sencillo.
`;

// ─── Remote Tool Proxy ───────────────────────────────────────────────────────

/**
 * Calls a tool on the remote Watch.dog PHP MCP server via HTTP.
 * The PHP server acts as the pure business logic layer.
 */
async function callRemoteTool(toolName, args = {}) {
  // Remove undefined args
  Object.keys(args).forEach((k) => args[k] === undefined && delete args[k]);

  if (!CONFIG.apiKey) {
    throw new Error(
      "WATCHDOG_API_KEY no está configurada. " +
        "Por favor configúrala en el archivo .env de la skill o como variable de entorno.",
    );
  }

  const response = await fetch(CONFIG.apiUrl, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      Accept: "application/json",
      Authorization: `Bearer ${CONFIG.apiKey}`,
    },
    body: JSON.stringify({
      jsonrpc: "2.0",
      id: `req_${Date.now()}`,
      method: "tools/call",
      params: { name: toolName, arguments: args },
    }),
  });

  if (!response.ok) {
    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
  }

  const data = await response.json();

  if (data.error) {
    throw new Error(`[${data.error.code}] ${data.error.message}`);
  }

  return data?.result?.content?.[0]?.text ?? "{}";
}

// ─── MCP Server Setup ────────────────────────────────────────────────────────

const server = new McpServer({
  name: "seeweb-uptime",
  version: "3.1.0",
});

// ── Prompts: Behavioral Instructions ─────────────────────────────────────────

server.registerPrompt(
  "seeweb-uptime",
  {
    description:
      "Activa el modo experto de SeeWeb Uptime. Proporciona el contexto de comportamiento para monitorizar infraestructura web, APIs y cron jobs.",
    argsSchema: {},
  },
  () => ({
    messages: [
      {
        role: "user",
        content: { type: "text", text: WATCHDOG_SYSTEM_PROMPT },
      },
    ],
  }),
);

// ── Tools: list_monitors ──────────────────────────────────────────────────────

server.registerTool(
  "list_monitors",
  {
    description: "Lista todos los monitores activos de uptime de la cuenta.",
    inputSchema: {
      status: z
        .enum(["up", "down", "paused", "all"])
        .optional()
        .describe("Opcional. Filtrar por status."),
    },
  },
  async ({ status }) => {
    const text = await callRemoteTool("list_monitors", { status });
    return {
      content: [
        {
          type: "text",
          text: `[Datos obtenidos. Preséntalos como una tabla markdown amigable usando emojis 🟢🔴⏸️]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: create_monitor ─────────────────────────────────────────────────────

server.registerTool(
  "create_monitor",
  {
    description:
      "Crea un Monitor Activo de Uptime que consulta periódicamente una URL, IP o puerto.",
    inputSchema: {
      name: z.string().describe("Nombre amigable del monitor"),
      url: z.string().describe("URL o hostname a monitorear"),
      type: z
        .enum(["http", "keyword", "ping", "port"])
        .describe("Tipo de verificación"),
      interval: z
        .number()
        .int()
        .positive()
        .describe("Intervalo de verificación en segundos (ej. 60)"),
      keyword: z
        .string()
        .optional()
        .describe(
          "Solo requerido si type=keyword. Cadena a buscar en el HTML.",
        ),
    },
  },
  async ({ name, url, type, interval, keyword }) => {
    const text = await callRemoteTool("create_monitor", {
      name,
      url,
      type,
      interval,
      keyword,
    });
    return {
      content: [
        {
          type: "text",
          text: `[Transmite este resultado JSON al usuario de forma entusiasta y celebra el éxito]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: get_monitor_status ─────────────────────────────────────────────────

server.registerTool(
  "get_monitor_status",
  {
    description:
      "Obtiene el estado de salud y eventos recientes de un monitor específico.",
    inputSchema: {
      monitor_id: z
        .number()
        .int()
        .positive()
        .optional()
        .describe("ID numérico del monitor"),
      name: z
        .string()
        .optional()
        .describe("Nombre exacto (case-sensitive) en caso de no saber el ID"),
    },
  },
  async ({ monitor_id, name }) => {
    if (!monitor_id && !name)
      throw new Error("Debes proveer monitor_id o name.");
    const text = await callRemoteTool("get_monitor_status", {
      monitor_id,
      name,
    });
    return {
      content: [
        {
          type: "text",
          text: `[Genera un resumen verbal del estado actual usando estos datos JSON]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: list_watchdogs ─────────────────────────────────────────────────────

server.registerTool(
  "list_watchdogs",
  {
    description:
      "Lista todos los Watchdogs (Heartbeats / Cron Jobs pasivos) de la cuenta.",
    inputSchema: {
      status: z
        .enum(["healthy", "missing", "all"])
        .optional()
        .describe("Opcional. Filtrar por estado de salud."),
    },
  },
  async ({ status }) => {
    const text = await callRemoteTool("list_watchdogs", { status });
    return {
      content: [
        {
          type: "text",
          text: `[Datos obtenidos. Preséntalos al usuario en una sola tabla Markdown con emojis de salud]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: create_watchdog ────────────────────────────────────────────────────

server.registerTool(
  "create_watchdog",
  {
    description:
      "Crea un Watchdog Pasivo (Cron Job / Heartbeat). Watch.dog esperará recibir un ping periódico.",
    inputSchema: {
      name: z.string().describe("Nombre del cron job o tarea programada"),
      type: z
        .enum(["email", "post", "get"])
        .describe("Método de recepción del ping"),
      interval: z
        .number()
        .int()
        .positive()
        .describe(
          "Margen de gracia en segundos. Si no llega un ping en este tiempo, se marca como caído.",
        ),
    },
  },
  async ({ name, type, interval }) => {
    const text = await callRemoteTool("create_watchdog", {
      name,
      type,
      interval,
    });
    return {
      content: [
        {
          type: "text",
          text: `[REGLA CRÍTICA: Debes enviarle al usuario la endpoint_url y darle instrucciones exactas de cómo hacer ping usando estos datos]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: get_watchdog_status ────────────────────────────────────────────────

server.registerTool(
  "get_watchdog_status",
  {
    description:
      "Obtiene el estado de salud (healthy/missing_ping) de un Watchdog y la información del último ping.",
    inputSchema: {
      watchdog_id: z
        .number()
        .int()
        .positive()
        .optional()
        .describe("ID numérico del watchdog"),
      name: z
        .string()
        .optional()
        .describe("Nombre exacto (case-sensitive) en caso de no saber el ID"),
    },
  },
  async ({ watchdog_id, name }) => {
    if (!watchdog_id && !name)
      throw new Error("Debes proveer watchdog_id o name.");
    const text = await callRemoteTool("get_watchdog_status", {
      watchdog_id,
      name,
    });
    return {
      content: [
        {
          type: "text",
          text: `[Resumen del estado del watchdog provisto por JSON]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: delete_monitor ─────────────────────────────────────────────────────
server.registerTool(
  "delete_monitor",
  {
    description:
      "Elimina un Monitor Activo de manera irreversible. REQUIERE CONFIRMACIÓN PREVIA TRIBUTARIA AL USUARIO.",
    inputSchema: {
      monitor_id: z.number().int().positive().optional(),
      name: z.string().optional(),
    },
  },
  async ({ monitor_id, name }) => {
    const text = await callRemoteTool("delete_monitor", { monitor_id, name });
    return {
      content: [
        {
          type: "text",
          text: `[Informa al usuario sobre el éxito de esta eliminación basándote en este JSON]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: delete_watchdog ────────────────────────────────────────────────────
server.registerTool(
  "delete_watchdog",
  {
    description:
      "Elimina un Watchdog Pasivo irreversiblemente. REQUIERE CONFIRMACIÓN PREVIA TRIBUTARIA AL USUARIO.",
    inputSchema: {
      watchdog_id: z.number().int().positive().optional(),
      name: z.string().optional(),
    },
  },
  async ({ watchdog_id, name }) => {
    const text = await callRemoteTool("delete_watchdog", { watchdog_id, name });
    return {
      content: [
        {
          type: "text",
          text: `[Informa al usuario sobre el éxito de esta eliminación basándote en este JSON]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: pause_monitor ──────────────────────────────────────────────────────
server.registerTool(
  "pause_monitor",
  {
    description: "Pausa temporalmente un Monitor Activo.",
    inputSchema: {
      monitor_id: z.number().int().positive().optional(),
      name: z.string().optional(),
    },
  },
  async ({ monitor_id, name }) => {
    const text = await callRemoteTool("pause_monitor", { monitor_id, name });
    return {
      content: [
        {
          type: "text",
          text: `[Monitor pausado exitosamente. Confírmaselo al usuario]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: resume_monitor ─────────────────────────────────────────────────────
server.registerTool(
  "resume_monitor",
  {
    description: "Reanuda un Monitor Activo previamente pausado.",
    inputSchema: {
      monitor_id: z.number().int().positive().optional(),
      name: z.string().optional(),
    },
  },
  async ({ monitor_id, name }) => {
    const text = await callRemoteTool("resume_monitor", { monitor_id, name });
    return {
      content: [
        {
          type: "text",
          text: `[Monitor reanudado exitosamente. Confírmaselo al usuario]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: get_monitor_uptime_history ─────────────────────────────────────────
server.registerTool(
  "get_monitor_uptime_history",
  {
    description:
      "Obtiene la matriz de Uptime/Disponibilidad de un monitor para los últimos X días.",
    inputSchema: {
      monitor_id: z.number().int().positive().optional(),
      name: z.string().optional(),
      days: z
        .number()
        .int()
        .positive()
        .optional()
        .describe("Cantidad de días a recuperar (por defecto 30)"),
    },
  },
  async ({ monitor_id, name, days }) => {
    const text = await callRemoteTool("get_monitor_uptime_history", {
      monitor_id,
      name,
      days,
    });
    return {
      content: [
        {
          type: "text",
          text: `[Analiza esta matriz JSON y preséntale un resumen verbal claro y amigable al usuario]:\n${text}`,
        },
      ],
    };
  },
);

// ── Tools: update_tracker_page ────────────────────────────────────────────────
server.registerTool(
  "update_tracker_page",
  {
    description:
      "Configura la Página Pública de Estado (Tracker Page) para mostrar monitores de cara al público.",
    inputSchema: {
      page_url: z
        .string()
        .optional()
        .describe("Alias o slug público para la empresa (ej. miempresa)."),
      monitors: z
        .array(z.string())
        .optional()
        .describe(
          "Array con nombres o IDs de los monitores que quieres hacer públicos.",
        ),
    },
  },
  async ({ page_url, monitors }) => {
    const text = await callRemoteTool("update_tracker_page", {
      page_url,
      monitors,
    });
    return {
      content: [
        {
          type: "text",
          text: `[Página de estado configurada exitosamente según este JSON. Infórmaselo al usuario y dile que puede visitar el enlace público de su Tracker Page]:\n${text}`,
        },
      ],
    };
  },
);

// ─── Start Server ────────────────────────────────────────────────────────────

const transport = new StdioServerTransport();
await server.connect(transport);
