---
name: security-dashboard
description: Real-time security monitoring dashboard for OpenClaw and Linux server infrastructure. Monitors gateway status, network security, public exposure, system updates, SSH access, TLS certificates, and resource usage.
---

# Security Dashboard Skill

Real-time security monitoring dashboard for OpenClaw and Linux server infrastructure.

## Features

- **OpenClaw Security:** Gateway status, binding, authentication, sessions, version tracking
- **Network Security:** Tailscale status, public ports, firewall, active connections
- **Public Exposure:** Port binding analysis, dashboard security, exposure level assessment
- **System Security:** Updates, uptime, load, failed login attempts
- **SSH & Access:** Password auth status, fail2ban, banned IPs, active sessions
- **Certificates & TLS:** Caddy status, TLS configuration, WireGuard encryption
- **Resource Security:** CPU/memory/disk usage, config file permissions

## Installation

### 1. Install the Skill

```bash
cd /root/clawd/skills/security-dashboard
sudo ./scripts/install.sh
```

This will:
- **Ask user preference:** Run as dedicated user (recommended) or root
- Create `openclaw-dashboard` user with limited sudo privileges (if non-root)
- Create systemd service with security hardening
- Configure localhost binding (127.0.0.1 only)
- Start the dashboard on port 18791
- Enable auto-start on boot

**Security Note:** Running as a dedicated user with limited sudo is recommended. The dashboard only needs sudo for security checks (fail2ban, firewall, systemctl status) - not full root access.

### 2. Access the Dashboard

**Localhost only (secure by default):**

Via SSH port forwarding:
```bash
ssh -L 18791:localhost:18791 root@YOUR_SERVER_IP
```

Then visit: http://localhost:18791

## Usage

### Start/Stop/Restart

```bash
sudo systemctl start security-dashboard
sudo systemctl stop security-dashboard
sudo systemctl restart security-dashboard
```

### Check Status

```bash
sudo systemctl status security-dashboard
```

### View Logs

```bash
sudo journalctl -u security-dashboard -f
```

### API Endpoint

Get raw security metrics:
```bash
curl http://localhost:18791/api/security | jq
```

## Security Hardening

The dashboard follows security best practices to minimize attack surface:

### Dedicated User (Recommended)
The install script creates a `openclaw-dashboard` user with **limited sudo privileges**:
- ✅ No shell access (`/bin/false`)
- ✅ No home directory
- ✅ Only specific sudo commands allowed (fail2ban, firewall, systemctl status)
- ✅ Cannot execute arbitrary commands

### Systemd Hardening
Service runs with security restrictions:
```ini
NoNewPrivileges=true      # Cannot escalate privileges
PrivateTmp=true          # Isolated tmp directory
ProtectSystem=strict     # Read-only filesystem except skill dir
ProtectHome=true         # No access to /home
ReadWritePaths=...       # Only skill directory is writable
Restart=on-failure       # Restart only on crashes (not always)
```

### Network Binding
- **Default:** `127.0.0.1` (localhost only)
- Not accessible from network without SSH tunnel or VPN
- No public exposure risk

### Running as Root (Not Recommended)
If you choose `root` during install:
- ⚠️ Full system access if compromised
- ⚠️ No privilege separation
- ⚠️ Only suitable for trusted, isolated environments

Use the dedicated user option for production deployments.

## Configuration

### Change Port

Edit `/root/clawd/skills/security-dashboard/server.js`:
```javascript
const PORT = 18791; // Change this
```

Then restart:
```bash
sudo systemctl restart security-dashboard
```

### Change Binding

**Default:** `127.0.0.1` (localhost only - secure)  
**Alternative:** `0.0.0.0` (all interfaces - only with Tailscale!)

Edit `server.js` line 445:
```javascript
server.listen(PORT, '127.0.0.1', () => {
  // Change '127.0.0.1' to '0.0.0.0' if needed
});
```

⚠️ **Security Warning:** Only bind to `0.0.0.0` if behind Tailscale or firewall!

### Customize Metrics

Add custom checks in `server.js`:
- `getOpenClawMetrics()` - OpenClaw-specific metrics
- `getNetworkMetrics()` - Network security
- `getSystemMetrics()` - System-level checks
- `getPublicExposure()` - Port/binding analysis

## Dashboard Sections

### 🦞 OpenClaw Security
- Gateway running/stopped status
- Binding configuration (loopback/public)
- Auth token length and mode
- Active sessions + subagents
- Skills count
- Current version + update availability

### 🌐 Network Security
- Tailscale connection status + IP
- Public ports count
- Firewall status (UFW/firewalld)
- Active TCP connections

### 🌍 Public Exposure
- Exposure level (Excellent/Minimal/Warning/High)
- Public port details (service names)
- Kanban board binding
- Security dashboard binding
- OpenClaw gateway binding
- Tailscale active/inactive
- Security recommendations

### 🖥️ System Security
- Updates available
- Server uptime
- Load average
- Failed SSH logins (24h)
- Root processes count

### 🔑 SSH & Access Control
- SSH service status
- Password authentication (enabled/disabled)
- fail2ban status
- Banned IPs count
- Active SSH sessions

### 📜 Certificates & TLS
- Caddy status
- Public TLS enabled/disabled
- Tailscale WireGuard encryption

### 📊 Resource Security
- CPU usage percentage
- Memory usage percentage
- Disk usage percentage
- Config file permissions (should be 600)

## Security Alerts

Dashboard generates real-time alerts:

**Critical (Red):**
- Weak gateway token (< 32 chars)
- SSH password authentication enabled
- Insecure config permissions (not 600)
- **Firewall inactive** (UFW/firewalld not running)
- **fail2ban inactive** (SSH brute-force protection disabled)

**Warning (Yellow):**
- Tailscale disconnected
- 20+ system updates available
- 10+ failed login attempts in 24h
- Disk > 80% full

**Info (Blue):**
- Gateway exposed without Tailscale
- Non-standard configurations

## Integration Points

### Morning Briefing
Add security status to morning report:
```bash
curl -s http://localhost:18791/api/security | jq '.status'
```

### Heartbeat Checks
Monitor for critical alerts:
```bash
curl -s http://localhost:18791/api/security | \
  jq '.alerts[] | select(.level == "critical")'
```

### Alerting Integration
Pipe alerts to notification systems:
```bash
./scripts/check-alerts.sh | xargs -I {} notify-send "Security Alert" "{}"
```

## Architecture

**Backend:** Node.js HTTP server  
**Frontend:** Vanilla JavaScript (no frameworks)  
**Port:** 18791 (configurable)  
**Binding:** 127.0.0.1 (localhost only)  
**Service:** systemd unit  

**Files:**
- `server.js` - Main backend (metrics collection + API)
- `public/index.html` - Dashboard UI
- `lib/` - Shared utilities (if needed)

## Dependencies

- Node.js (v18+)
- `systemctl` - Service management
- `ss` - Socket statistics
- `ufw` or `firewalld` - Firewall check
- `tailscale` - VPN status (optional)
- `fail2ban` - Ban tracking (optional)
- `openclaw` - Gateway monitoring

All dependencies are standard Linux utilities except OpenClaw.

## Troubleshooting

### Dashboard not loading

1. Check service status:
   ```bash
   sudo systemctl status security-dashboard
   ```

2. Check logs:
   ```bash
   sudo journalctl -u security-dashboard -n 50
   ```

3. Verify port is listening:
   ```bash
   ss -tlnp | grep 18791
   ```

4. Test API directly:
   ```bash
   curl http://localhost:18791/api/security
   ```

### Gateway Status "Unknown"

- Verify OpenClaw gateway is running:
  ```bash
  pgrep -f openclaw-gateway
  ```

- Check OpenClaw config exists:
  ```bash
  cat ~/.openclaw/openclaw.json
  ```

### Metrics showing "Unknown"

- Commands may require sudo permissions
- Check script execution permissions
- Verify paths exist (sessions, skills, etc.)

## Uninstall

```bash
sudo systemctl stop security-dashboard
sudo systemctl disable security-dashboard
sudo rm /etc/systemd/system/security-dashboard.service
sudo systemctl daemon-reload
```

Then remove skill directory:
```bash
rm -rf /root/clawd/skills/security-dashboard
```

## Publishing

To publish to ClawdHub:
```bash
clawdhub publish security-dashboard
```

## License

MIT

## Author

Created by Erdma for Brian Christner's infrastructure monitoring.
