#!/bin/bash
# 诊断脚本

WORKSPACE="/root/clawd"

echo "🔍 安全记忆系统全面诊断"
echo ""

echo "======= 系统环境检查 ======="
echo "Python版本:"
python3 --version 2>/dev/null || echo "❌ Python3未安装"

echo ""
echo "Git版本:"
git --version 2>/dev/null || echo "❌ Git未安装"

echo ""
echo "======= 组件健康检查 ======="

# 检查百度Embedding
echo "百度Embedding系统:"
python3 -c "
import sys
sys.path.append('/root/clawd/skills/memory-baidu-embedding-db')
try:
    from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB
    db = MemoryBaiduEmbeddingDB()
    stats = db.get_statistics()
    print(f'  ✅ 连接正常 - {stats[\"total_memories\"]} 条记忆')
except ImportError:
    print('  ❌ 模块未找到 - 检查路径')
except Exception as e:
    print(f'  ❌ 连接失败 - {str(e)}')
"

# 检查Git Notes
echo ""
echo "Git Notes系统:"
if command -v git >/dev/null 2>&1; then
    if [ -d "$WORKSPACE/.git" ]; then
        echo "  ✅ Git可用且仓库已初始化"
        # 测试Git Notes功能
        python3 /root/clawd/skills/git-notes-memory/memory.py -p "$WORKSPACE" sync --start >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            echo "  ✅ Git Notes同步功能正常"
        else
            echo "  ❌ Git Notes同步功能异常"
        fi
    else
        echo "  ❌ Git仓库未初始化"
        echo "  💡 运行: secure-memory fix git"
    fi
else
    echo "  ❌ Git未安装"
    echo "  💡 请安装Git"
fi

# 检查文件系统
echo ""
echo "文件系统:"
FILES=("$WORKSPACE/MEMORY.md" "$WORKSPACE/SESSION-STATE.md" "$WORKSPACE/memory/$(date +%Y-%m-%d).md")
for file in "${FILES[@]}"; do
    if [ -f "$file" ]; then
        size=$(stat -c%s "$file" 2>/dev/null || echo 0)
        echo "  ✅ $(basename $file) (大小: ${size}字节)"
    else
        echo "  ❌ $(basename $file) (缺失)"
    fi
done

# 检查API配置
echo ""
echo "API配置:"
if [ -n "$BAIDU_API_STRING" ] || [ -n "$BAIDU_API_KEY" ]; then
    echo "  ✅ 百度API已配置"
else
    echo "  ⚠️ 百度API未配置"
    echo "  💡 运行: secure-memory configure baidu"
fi

echo ""
echo "======= 安全检查 ======="
echo "数据存储: 本地设备"
echo "网络传输: 无 (完全离线)"
echo "访问控制: 本地访问"
echo "隐私保护: 最高级别"

echo ""
echo "======= 建议 ======="
echo "1. 如果Git Notes功能异常，运行: secure-memory fix git"
echo "2. 如果API未配置，运行: secure-memory configure baidu"
echo "3. 定期运行: secure-memory status 检查系统状态"
echo "4. 使用: secure-memory stats 查看详细统计"

echo ""
echo "诊断完成。如需帮助，运行: secure-memory help"