#!/bin/bash
# 记忆系统健康检查脚本
# memory_health_check.sh

echo "🏥 记忆系统健康检查开始..."

# 初始化计分器
PASS_COUNT=0
TOTAL_CHECKS=0

# 检查1: 环境变量
echo "🔍 检查1: 环境变量配置"
TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
if [ -n "$BAIDU_API_STRING" ]; then
    echo "   ✅ BAIDU_API_STRING 已设置"
    PASS_COUNT=$((PASS_COUNT + 1))
else
    echo "   ❌ BAIDU_API_STRING 未设置"
fi

TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
if [ -n "$BAIDU_SECRET_KEY" ]; then
    echo "   ✅ BAIDU_SECRET_KEY 已设置"
    PASS_COUNT=$((PASS_COUNT + 1))
else
    echo "   ❌ BAIDU_SECRET_KEY 未设置"
fi

# 检查2: 核心文件存在
echo "🔍 检查2: 核心文件存在性"
CORE_FILES=(
    "/root/clawd/memory_bootstrap.sh"
    "/root/clawd/MEMORY_BOOTSTRAP.md"
    "/root/clawd/hooks/hook_memory_bootstrap.sh"
    "/root/clawd/MEMORY.md"
)

for file in "${CORE_FILES[@]}"; do
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if [ -f "$file" ]; then
        echo "   ✅ $(basename "$file") 存在"
        PASS_COUNT=$((PASS_COUNT + 1))
    else
        echo "   ❌ $file 不存在"
    fi
done

# 检查3: 可执行权限
echo "🔍 检查3: 可执行权限"
EXEC_FILES=(
    "/root/clawd/memory_bootstrap.sh"
    "/root/clawd/memory_performance_monitor.sh"
    "/root/clawd/memory_maintenance.sh"
    "/root/clawd/hooks/hook_memory_bootstrap.sh"
)

for file in "${EXEC_FILES[@]}"; do
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if [ -x "$file" ]; then
        echo "   ✅ $file 可执行"
        PASS_COUNT=$((PASS_COUNT + 1))
    else
        echo "   ⚠️  $file 不可执行 (尝试修复)"
        chmod +x "$file" 2>/dev/null
        if [ $? -eq 0 ]; then
            echo "      📝 权限已修复"
            PASS_COUNT=$((PASS_COUNT + 1))
        fi
    fi
done

# 检查4: 技能组件
echo "🔍 检查4: 记忆系统组件"
COMPONENTS=(
    "memory-baidu-embedding-db"
    "git-notes-memory"
    "triple-memory"
)

for component in "${COMPONENTS[@]}"; do
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if [ -d "/root/clawd/skills/$component" ]; then
        echo "   ✅ $component 可用"
        PASS_COUNT=$((PASS_COUNT + 1))
    else
        echo "   ⚠️  $component 不可用"
    fi
done

# 检查5: 白名单技能
echo "🔍 检查5: 白名单技能"
WHITELIST_SKILLS=("ai-sql" "x-api" "oauth-helper")
for skill in "${WHITELIST_SKILLS[@]}"; do
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if [ -d "/root/clawd/skills/$skill" ] && [ -f "/root/clawd/skills/$skill/WHITELISTED.md" ]; then
        echo "   ✅ $skill 在白名单中"
        PASS_COUNT=$((PASS_COUNT + 1))
    elif [ -d "/root/clawd/skills/$skill" ]; then
        echo "   ⚠️  $skill 已安装但不在白名单"
    else
        echo "   ⚠️  $skill 未安装"
    fi
done

# 检查6: 记忆目录
echo "🔍 检查6: 记忆存储目录"
TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
if [ -d "/root/clawd/memory" ]; then
    echo "   ✅ 记忆目录存在"
    PASS_COUNT=$((PASS_COUNT + 1))
else
    echo "   ⚠️  记忆目录不存在 (尝试创建)"
    mkdir -p /root/clawd/memory
    if [ $? -eq 0 ]; then
        echo "      📝 目录已创建"
        PASS_COUNT=$((PASS_COUNT + 1))
    fi
fi

# 检查7: Git配置
echo "🔍 检查7: Git配置 (用于Git Notes)"
if command -v git >/dev/null 2>&1; then
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if git -C /root/clawd rev-parse 2>/dev/null; then
        echo "   ✅ Git仓库已初始化"
        PASS_COUNT=$((PASS_COUNT + 1))
    else
        echo "   ⚠️  Git仓库未初始化"
    fi
else
    echo "   ⚠️  Git未安装"
fi

# 输出健康检查结果
echo ""
echo "📊 健康检查结果: $PASS_COUNT/$TOTAL_CHECKS 通过"
SCORE=$((PASS_COUNT * 100 / TOTAL_CHECKS))

if [ $SCORE -ge 90 ]; then
    STATUS="🟢 优秀"
elif [ $SCORE -ge 70 ]; then
    STATUS="🟡 良好"
elif [ $SCORE -ge 50 ]; then
    STATUS="🟠 一般"
else
    STATUS="🔴 需要关注"
fi

echo "📈 整体健康度: $SCORE% ($STATUS)"

if [ $SCORE -ge 80 ]; then
    echo "✅ 记忆系统整体健康状况良好"
else
    echo "⚠️  建议运行维护脚本: /root/clawd/memory_maintenance.sh optimize"
fi

echo "🏥 记忆系统健康检查完成"