# 安全记忆系统栈升级与迁移指南

本文档介绍如何升级安全记忆系统栈以及从其他记忆系统迁移到本系统。

## 🔄 版本升级

### 从旧版本升级

```bash
# 通过ClawdHub升级
clawdhub update secure-memory-stack

# 或者手动升级
clawdhub install secure-memory-stack --force
```

### 升级前注意事项

1. **备份数据** - 升级前务必备份重要数据
```bash
# 备份MEMORY.md
cp /root/clawd/MEMORY.md ~/backup_MEMORY_$(date +%Y%m%d).md
```

2. **检查兼容性** - 确认新版本与现有配置兼容

3. **测试环境** - 建议先在测试环境中验证升级

### 升级后验证

```bash
# 验证系统状态
secure-memory diagnose

# 检查数据完整性
secure-memory stats
```

## 📤 数据迁移

### 从其他记忆系统迁移

#### 迁移JSON格式数据

如果您的数据是JSON格式，可以使用以下方法导入：

```bash
# 创建临时导入脚本
cat > /tmp/import_data.json << 'EOF'
[
  {
    "content": "记忆内容1",
    "tags": ["tag1", "tag2"],
    "importance": "high"
  },
  {
    "content": "记忆内容2", 
    "tags": ["tag3"],
    "importance": "normal"
  }
]
EOF

# 使用脚本导入
python3 -c "
import json
with open('/tmp/import_data.json') as f:
    data = json.load(f)
    
for item in data:
    content = item['content']
    tags = ','.join(item.get('tags', []))
    importance = item.get('importance', 'normal')
    print(f'secure-memory remember \"{content}\" --tags {tags} --importance {importance}')
"
```

#### 迁移文本文件

对于纯文本数据，可以批量导入：

```bash
# 假设文本文件每行一个记忆点
while read line; do
    secure-memory remember "$line" --tags imported --importance normal
done < your_data.txt
```

### 从LanceDB迁移

如果之前使用过LanceDB，可以导出现有数据：

```bash
# 导出现有记忆（如果可用）
python3 -c "
# 示例迁移脚本
import sys
sys.path.append('/root/clawd/skills/memory-baidu-embedding-db')
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

try:
    db = MemoryBaiduEmbedDB()
    memories = db.get_all_memories()
    for mem in memories:
        content = mem.get('content', '')
        if content:
            import subprocess
            subprocess.run(['secure-memory', 'remember', content, '--tags', 'migrated', '--importance', 'normal'])
except Exception as e:
    print(f'Migration failed: {e}')
"
```

## 📥 导入功能

### 批量导入

创建批量导入脚本：

```bash
# 创建批量导入函数
import_batch() {
    local file=$1
    local default_tags=${2:-"imported"}
    local default_importance=${3:-"normal"}
    
    while IFS= read -r line; do
        if [ -n "$line" ]; then
            secure-memory remember "$line" --tags "$default_tags" --importance "$default_importance"
        fi
    done < "$file"
}

# 使用示例
# import_batch my_memories.txt "personal,migrated" "normal"
```

### 从CSV导入

如果有CSV格式的数据：

```bash
# CSV格式: content,tags,importance
tail -n +2 your_data.csv | while IFS=, read -r content tags importance; do
    secure-memory remember "$content" --tags "$tags" --importance "$importance"
done
```

## 🔄 配置迁移

### 环境变量配置

将旧配置迁移到新系统：

```bash
# 检查当前配置
env | grep -i memory
env | grep -i baidu

# 设置新的环境变量
export BAIDU_API_KEY="your_new_api_key"
export BAIDU_SECRET_KEY="your_new_secret_key"
```

### 配置文件迁移

```bash
# 检查现有配置文件
ls -la /root/clawd/memory_config.json

# 如需调整配置，可直接编辑
# vim /root/clawd/memory_config.json
```

## 🧪 验证迁移

### 数据完整性检查

```bash
# 检查迁移后的数据
secure-memory stats

# 搜索特定内容验证
secure-memory search "migration test"
secure-memory search "imported"
```

### 功能验证

```bash
# 测试各项功能
secure-memory status
secure-memory diagnose

# 测试搜索功能
secure-memory search "test"

# 测试添加功能
secure-memory remember "Test successful" --tags test --importance normal
```

## 🚨 常见问题

### 迁移失败

如果迁移过程中出现问题：

1. **停止迁移** - 立即停止当前迁移过程
2. **数据备份** - 确保现有数据安全
3. **检查日志** - 查看错误信息
4. **重试或求助** - 根据错误信息重试或寻求支持

### 数据不一致

```bash
# 运行系统诊断
secure-memory diagnose

# 检查数据状态
secure-memory stats

# 修复可能的问题
secure-memory fix all
```

## 📋 迁移清单

### 准备阶段
- [ ] 备份现有数据
- [ ] 检查系统要求
- [ ] 准备迁移数据
- [ ] 测试环境验证

### 执行阶段
- [ ] 安装新系统
- [ ] 初始化配置
- [ ] 执行数据迁移
- [ ] 验证数据完整性

### 验证阶段
- [ ] 功能测试
- [ ] 搜索测试
- [ ] 性能评估
- [ ] 用户验收

## 📞 支持

如果在升级或迁移过程中遇到问题：

```bash
# 获取系统诊断信息
secure-memory diagnose > diagnostic_report.txt

# 检查系统状态
secure-memory status

# 查看帮助
secure-memory help
```

迁移完成后，记得清理临时文件并验证所有功能正常工作。