#!/usr/bin/env python3
"""
SearXNG Search Script
Performs web searches via self-hosted or public SearXNG instance
"""

import sys
import json
import urllib.request
import urllib.parse
import urllib.error
import os
from html.parser import HTMLParser

# Default to localhost, override with SEARXNG_URL environment variable
SEARXNG_URL = os.environ.get('SEARXNG_URL', 'http://127.0.0.1:8080')

class SearXNGParser(HTMLParser):
    """Parse SearXNG HTML results"""
    def __init__(self):
        super().__init__()
        self.results = []
        self.current_result = {}
        self.in_article = False
        self.in_h3 = False
        self.in_h3_link = False
        self.in_content = False
        self.capture_text = []
        
    def handle_starttag(self, tag, attrs):
        attrs_dict = dict(attrs)
        
        # Start of result
        if tag == 'article' and 'result' in attrs_dict.get('class', ''):
            self.in_article = True
            self.current_result = {}
        
        # Title container
        elif self.in_article and tag == 'h3':
            self.in_h3 = True
        
        # Title link
        elif self.in_h3 and tag == 'a':
            self.in_h3_link = True
            self.current_result['url'] = attrs_dict.get('href', '')
            self.capture_text = []
        
        # Content paragraph
        elif self.in_article and tag == 'p' and 'content' in attrs_dict.get('class', ''):
            self.in_content = True
            self.capture_text = []
    
    def handle_endtag(self, tag):
        if tag == 'article' and self.in_article:
            self.in_article = False
            if 'title' in self.current_result and 'url' in self.current_result:
                self.results.append(self.current_result.copy())
            self.current_result = {}
            
        elif tag == 'h3' and self.in_h3:
            self.in_h3 = False
            
        elif tag == 'a' and self.in_h3_link:
            self.in_h3_link = False
            # Clean up title text (remove extra whitespace)
            title = ' '.join(''.join(self.capture_text).split())
            self.current_result['title'] = title
            self.capture_text = []
            
        elif tag == 'p' and self.in_content:
            self.in_content = False
            # Clean up content text
            content = ' '.join(''.join(self.capture_text).split())
            self.current_result['content'] = content
            self.capture_text = []
    
    def handle_data(self, data):
        if self.in_h3_link or self.in_content:
            self.capture_text.append(data)


def search(query, categories=None, engines=None, language="en", num_results=10, bang=None):
    """
    Search SearXNG and return results
    
    Args:
        query: Search query string
        categories: Comma-separated categories (general, images, news, etc.)
        engines: Comma-separated engines to use
        language: Language code (default: en)
        num_results: Number of results to return (default: 10)
        bang: DuckDuckGo-style bang (!g, !w, !yt, etc.) for direct engine search
    
    Returns:
        dict with results
    """
    
    # If bang is provided, prepend it to the query
    if bang:
        if not bang.startswith('!'):
            bang = '!' + bang
        query = f"{bang} {query}"
    
    params = {
        'q': query,
        'language': language
    }
    
    if categories:
        params['categories'] = categories
    if engines:
        params['engines'] = engines
    
    url = f"{SEARXNG_URL}/search?{urllib.parse.urlencode(params)}"
    
    try:
        headers = {
            'User-Agent': 'Mozilla/5.0 (OpenClaw Agent)',
            'Accept': 'text/html'
        }
        
        req = urllib.request.Request(url, headers=headers)
        
        with urllib.request.urlopen(req, timeout=15) as response:
            html = response.read().decode('utf-8')
            
            # Parse HTML
            parser = SearXNGParser()
            parser.feed(html)
            
            results = parser.results[:num_results]
            
            return {
                'query': query,
                'number_of_results': len(results),
                'results': results
            }
            
    except urllib.error.HTTPError as e:
        return {'error': f'HTTP Error {e.code}: {e.reason}'}
    except urllib.error.URLError as e:
        return {'error': f'Connection error: {e.reason}'}
    except Exception as e:
        return {'error': f'Unexpected error: {str(e)}'}


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: search.py <query> [--categories CATS] [--engines ENGS] [--lang LANG] [--num N] [--bang BANG]")
        print("\nExamples:")
        print("  search.py 'OpenClaw agent'")
        print("  search.py 'python tutorial' --categories general --lang en --num 5")
        print("  search.py 'breaking news' --categories news")
        print("  search.py 'machine learning' --bang w  # Wikipedia search")
        print("  search.py 'cat videos' --bang yt       # YouTube search")
        print("\nEnvironment:")
        print(f"  SEARXNG_URL: {SEARXNG_URL}")
        sys.exit(1)
    
    query = sys.argv[1]
    categories = None
    engines = None
    language = "en"
    num_results = 10
    bang = None
    
    # Parse optional arguments
    i = 2
    while i < len(sys.argv):
        if sys.argv[i] == '--categories' and i + 1 < len(sys.argv):
            categories = sys.argv[i + 1]
            i += 2
        elif sys.argv[i] == '--engines' and i + 1 < len(sys.argv):
            engines = sys.argv[i + 1]
            i += 2
        elif sys.argv[i] == '--lang' and i + 1 < len(sys.argv):
            language = sys.argv[i + 1]
            i += 2
        elif sys.argv[i] == '--num' and i + 1 < len(sys.argv):
            num_results = int(sys.argv[i + 1])
            i += 2
        elif sys.argv[i] == '--bang' and i + 1 < len(sys.argv):
            bang = sys.argv[i + 1]
            i += 2
        else:
            i += 1
    
    result = search(query, categories, engines, language, num_results, bang)
    print(json.dumps(result, indent=2, ensure_ascii=False))
