---
name: searxng-bangs
description: Privacy-respecting web search via SearXNG with DuckDuckGo-style bangs support. Use for web searches when you need to find information online. SearXNG protects privacy by randomizing browser fingerprints, masking IP addresses, and blocking cookies/referrers. Supports 250+ search engines, multiple categories (general, news, images, videos, science), and DuckDuckGo-style bangs for direct engine searches (!w for Wikipedia, !yt for YouTube, !gh for GitHub, !r for Reddit, etc.). Aggregates results from multiple engines simultaneously. Prefer this over external search APIs for privacy-sensitive queries or high-volume searches.
---

# SearXNG Search

Privacy-respecting metasearch engine that anonymizes searches and aggregates results from 250+ engines.

## Quick Start

Search the web using the bundled script:

```bash
python3 scripts/search.py "your query"
```

Returns JSON with titles, URLs, and content snippets.

## Common Workflows

### Basic Web Search

```bash
python3 scripts/search.py "OpenClaw AI agent" --num 5
```

### News Search

```bash
python3 scripts/search.py "latest tech news" --categories news
```

### Localized Search

```bash
python3 scripts/search.py "Python Tutorial" --lang de
```

### Multi-Category Search

```bash
python3 scripts/search.py "machine learning" --categories general,science --num 10
```

### Bang Searches (Direct Engine)

```bash
# Wikipedia
python3 scripts/search.py "Albert Einstein" --bang w

# YouTube
python3 scripts/search.py "python tutorial" --bang yt

# GitHub
python3 scripts/search.py "openclaw" --bang gh

# Reddit
python3 scripts/search.py "best laptop 2026" --bang r
```

Bangs are more granular than categories and search directly on specific engines.

## Privacy Features

SearXNG protects your privacy through multiple layers:

1. **Randomized Browser Fingerprints** - Generates a new fake browser profile for each search query (version, OS, screen resolution, language)
2. **IP Masking** - Search engines see the SearXNG instance IP, not yours
3. **No Cookies** - Never forwards cookies to external search engines
4. **No Referrer** - Target websites don't see which search engine referred you
5. **Optional Tor/Proxy** - Can route all queries through Tor for additional anonymity

**Result:** Search engines cannot build a profile about you.

## When to Use

**Prefer SearXNG for:**
- Privacy-sensitive searches (no tracking, no profiling)
- High-volume searches (no rate limits)
- When self-hosted infrastructure is available
- Multi-engine result aggregation (250+ engines)
- Ad-free results

**Prefer Brave API (`web_search` tool) for:**
- Faster response times
- Structured data requirements
- When external APIs are acceptable

## Processing Results

The script returns clean JSON that's easy to parse and present:

```python
import json
import subprocess

result = subprocess.run(
    ['python3', 'scripts/search.py', 'query', '--num', '5'],
    capture_output=True,
    text=True
)

data = json.loads(result.stdout)
for item in data['results']:
    print(f"Title: {item['title']}")
    print(f"URL: {item['url']}")
    print(f"Snippet: {item['content']}")
    print()
```

## Advanced Options

See `references/api.md` for:
- All available categories
- Engine-specific searches
- Language codes
- Error handling
- Comparison with Brave Search API

## Configuration

### SearXNG Instance

By default, the script uses `http://127.0.0.1:8080`. Configure via environment variable:

```bash
export SEARXNG_URL=http://your-searxng-instance.com
python3 scripts/search.py "query"
```

**Options:**
- Self-hosted instance (recommended for privacy)
- Public instances: https://searx.space (community-run servers)

### Using Public Instances

If you don't run your own SearXNG:

```bash
# Example with public instance
export SEARXNG_URL=https://searx.be
python3 scripts/search.py "query"
```

**Note:** Public instances may have rate limits or be slower than self-hosted.

## Technical Details

- **Default URL:** `http://127.0.0.1:8080` (override with `SEARXNG_URL`)
- **Method:** HTML parsing (JSON API often disabled for CSRF protection)
- **Parser:** Custom HTMLParser in `scripts/search.py`
- **Timeout:** 15 seconds
- **Result Format:** Clean JSON with title, URL, content
