#!/usr/bin/env python3
"""
Scalekit Auth Helper - Token Management for AI Agents
Handles OAuth token retrieval, refresh, and authorization via Scalekit.
"""

import os
import json
import sys
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables
skill_dir = Path(__file__).parent
load_dotenv(skill_dir / ".env")

# Import Scalekit SDK
try:
    import scalekit.client
except ImportError:
    print("ERROR: scalekit-sdk-python not installed. Run: pip3 install -r requirements.txt", file=sys.stderr)
    sys.exit(1)


class ConfigurationError(Exception):
    """Raised when Scalekit or connection is not configured"""
    pass


class AuthorizationError(Exception):
    """Raised when user needs to authorize the connection"""
    pass


def get_scalekit_client():
    """Initialize and return Scalekit client"""
    client_id = os.getenv("SCALEKIT_CLIENT_ID")
    client_secret = os.getenv("SCALEKIT_CLIENT_SECRET")
    env_url = os.getenv("SCALEKIT_ENV_URL")
    
    if not all([client_id, client_secret, env_url]):
        raise ConfigurationError(
            "Scalekit credentials not configured. Please create skills/scalekit-auth/.env with:\n"
            "SCALEKIT_CLIENT_ID=...\n"
            "SCALEKIT_CLIENT_SECRET=...\n"
            "SCALEKIT_ENV_URL=...\n\n"
            "Get credentials from: https://app.scalekit.com (Developers → Settings → API Credentials)"
        )
    
    return scalekit.client.ScalekitClient(
        client_id=client_id,
        client_secret=client_secret,
        env_url=env_url
    )


def load_connections():
    """Load connections.json configuration"""
    connections_file = skill_dir / "connections.json"
    
    if not connections_file.exists():
        return {}
    
    with open(connections_file, 'r') as f:
        return json.load(f)


def save_connections(connections):
    """Save connections.json configuration"""
    connections_file = skill_dir / "connections.json"
    
    with open(connections_file, 'w') as f:
        json.dump(connections, f, indent=2)


def get_agent_identifier():
    """Get agent identifier from IDENTITY.md or fallback to 'agent'"""
    identity_file = Path.home() / ".openclaw/workspace/IDENTITY.md"
    
    if identity_file.exists():
        content = identity_file.read_text()
        # Extract name from "**Name:** <name>" line
        for line in content.split('\n'):
            if line.strip().startswith('- **Name:**') or line.strip().startswith('**Name:**'):
                name = line.split(':', 1)[1].strip()
                return name.lower()
    
    return "agent"


def configure_connection(service_name, connection_name):
    """
    Store connection configuration for a service
    
    Args:
        service_name: Human-readable service name (e.g., "gmail")
        connection_name: Scalekit connection name (e.g., "gmail_u3134a")
    """
    connections = load_connections()
    identifier = get_agent_identifier()
    
    connections[service_name] = {
        "connection_name": connection_name,
        "identifier": identifier
    }
    
    save_connections(connections)
    print(f"✅ Configured {service_name} with connection_name: {connection_name}")


def get_token(service_name):
    """
    Get fresh OAuth access token for a service
    
    Args:
        service_name: Service identifier (e.g., "gmail", "slack")
    
    Returns:
        str: Fresh OAuth access token
    
    Raises:
        ConfigurationError: Service or Scalekit not configured
        AuthorizationError: User needs to authorize (includes auth link)
    """
    # Load connection config
    connections = load_connections()
    
    if service_name not in connections:
        raise ConfigurationError(
            f"{service_name} not configured.\n\n"
            f"Please:\n"
            f"1. Create {service_name} connection in Scalekit dashboard\n"
            f"2. Configure OAuth (client ID/secret from provider)\n"
            f"3. Run: configure_connection('{service_name}', 'connection_name_from_scalekit')\n\n"
            f"Or tell the agent: 'Configure {service_name}. Connection name is <name>'"
        )
    
    connection_config = connections[service_name]
    connection_name = connection_config["connection_name"]
    identifier = connection_config["identifier"]
    
    # Initialize Scalekit client
    try:
        client = get_scalekit_client()
    except ConfigurationError as e:
        raise e
    
    actions = client.actions
    
    # Get or create connected account
    try:
        response = actions.get_or_create_connected_account(
            connection_name=connection_name,
            identifier=identifier
        )
    except Exception as e:
        raise ConfigurationError(
            f"Failed to get connected account for {service_name}.\n"
            f"Connection name: {connection_name}\n"
            f"Identifier: {identifier}\n"
            f"Error: {str(e)}\n\n"
            f"Verify connection exists in Scalekit dashboard."
        )
    
    connected_account = response.connected_account
    
    # Check if authorization is needed
    if connected_account.status != "ACTIVE":
        # Generate authorization link
        try:
            link_response = actions.get_authorization_link(
                connection_name=connection_name,
                identifier=identifier
            )
            auth_link = link_response.link
        except Exception as e:
            raise AuthorizationError(
                f"Failed to generate authorization link for {service_name}.\n"
                f"Error: {str(e)}"
            )
        
        raise AuthorizationError(
            f"⚠️ Authorization needed for {service_name}!\n\n"
            f"🔗 Authorization link (expires in 1 minute):\n"
            f"```\n{auth_link}\n```\n\n"
            f"Copy and paste the link in your browser to authorize.\n"
            f"After authorizing, try your request again."
        )
    
    # Extract and return access token
    try:
        tokens = connected_account.authorization_details["oauth_token"]
        access_token = tokens["access_token"]
        return access_token
    except (KeyError, TypeError) as e:
        raise ConfigurationError(
            f"Failed to extract access token for {service_name}.\n"
            f"Connected account status: {connected_account.status}\n"
            f"Error: {str(e)}"
        )


def main():
    """CLI wrapper for get_token"""
    if len(sys.argv) < 2:
        print("Usage: python3 scalekit_helper.py <service_name>", file=sys.stderr)
        print("Example: python3 scalekit_helper.py gmail", file=sys.stderr)
        sys.exit(1)
    
    service_name = sys.argv[1]
    
    try:
        token = get_token(service_name)
        print(token)
    except (ConfigurationError, AuthorizationError) as e:
        print(str(e), file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Unexpected error: {str(e)}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
