#!/usr/bin/env python3
"""
Agent-friendly wrapper for getting tokens
Handles authorization links by sending them via messaging channels
"""

import sys
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent))

from scalekit_helper import get_token, ConfigurationError, AuthorizationError


def get_token_for_agent(service_name, send_via_message=True):
    """
    Get token with agent-friendly error handling
    
    If authorization is needed and send_via_message=True,
    returns a dict with status and auth_link for the agent to send
    """
    try:
        token = get_token(service_name)
        return {"status": "success", "token": token}
    except AuthorizationError as e:
        error_msg = str(e)
        # Extract the link from the error message
        if "```" in error_msg:
            link = error_msg.split("```")[1].strip()
        else:
            link = None
        
        return {
            "status": "needs_auth",
            "service": service_name,
            "auth_link": link,
            "message": f"⚠️ Authorization needed for {service_name}!\n\n🔗 Copy this link (expires in 1 minute):\n```\n{link}\n```\n\nPaste it in your browser to authorize."
        }
    except ConfigurationError as e:
        return {
            "status": "error",
            "service": service_name,
            "message": str(e)
        }


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python3 agent_wrapper.py <service_name>", file=sys.stderr)
        sys.exit(1)
    
    service_name = sys.argv[1]
    result = get_token_for_agent(service_name)
    
    if result["status"] == "success":
        print(result["token"])
    else:
        print(result["message"], file=sys.stderr)
        sys.exit(1)
