---
name: scalekit-auth
description: Secure OAuth token management via Scalekit. Handles token storage, refresh, and retrieval for third-party services (Gmail, Slack, GitHub, etc.). Never stores tokens locally - always fetches fresh tokens from Scalekit.
homepage: https://scalekit.com
metadata:
  openclaw:
    requires:
      bins: ["python3"]
    install:
      - id: python-deps
        kind: exec
        command: "pip3 install scalekit-sdk-python python-dotenv requests"
        label: "Install Python dependencies"
---

# Scalekit Auth - Secure Token Management

Centralized OAuth token management for AI agents. No local token storage, automatic refresh, multi-service support.

## Why Use This?

**Problem:** OAuth tokens scattered across config files, no refresh logic, security risks.

**Solution:** Scalekit handles all token lifecycle:
- ✅ Secure cloud storage (never stored locally)
- ✅ Automatic token refresh
- ✅ Multi-service support (Gmail, Slack, Notion, GitHub, etc.)
- ✅ Always returns fresh, valid tokens

## Installation

### 1. Install Skill

```bash
clawhub install scalekit-auth
cd skills/scalekit-auth
pip3 install -r requirements.txt
```

### 2. Get Scalekit Credentials

1. Sign up at [scalekit.com](https://scalekit.com)
2. Go to Dashboard → Developers → Settings → API Credentials
3. Copy:
   - Client ID
   - Client Secret
   - Environment URL

### 3. Configure Credentials

Create `skills/scalekit-auth/.env`:

```bash
SCALEKIT_CLIENT_ID=your_client_id_here
SCALEKIT_CLIENT_SECRET=your_client_secret_here
SCALEKIT_ENV_URL=https://your-env.scalekit.com
```

**Or** let the agent ask you on first use.

## Setting Up a Service (e.g., Gmail)

### Step 1: Create Connection in Scalekit Dashboard

1. Go to Scalekit Dashboard → Connections → Add Connection
2. Select provider (e.g., Gmail/Google)
3. Configure OAuth:
   - Get Client ID/Secret from Google Cloud Console
   - Set Redirect URI (provided by Scalekit)
4. **Copy the `connection_name`** (e.g., `gmail_u3134a`)

### Step 2: Register with Agent

Tell the agent:
```
"Configure Gmail for Scalekit. Connection name is gmail_u3134a"
```

Agent stores it in `connections.json`:
```json
{
  "gmail": {
    "connection_name": "gmail_u3134a",
    "identifier": "mess"
  }
}
```

### Step 3: Authorize

First API call will prompt:
```
Authorization needed for Gmail.
Link: https://scalekit.com/auth/... (expires in 1 minute!)
```

Click link → authorize → done!

## Usage

### From Agent Skills

```python
#!/usr/bin/env python3
import sys
sys.path.append('./skills/scalekit-auth')
from scalekit_helper import get_token

# Get fresh token for any service
access_token = get_token("gmail")

# Use it immediately
headers = {"Authorization": f"Bearer {access_token}"}
response = requests.get("https://gmail.googleapis.com/gmail/v1/users/me/messages", headers=headers)
```

### From Shell Scripts

```bash
# Get token via CLI wrapper
TOKEN=$(python3 skills/scalekit-auth/get_token.py gmail)

# Use in API call
curl -H "Authorization: Bearer $TOKEN" \
  https://gmail.googleapis.com/gmail/v1/users/me/messages
```

## Configuration Files

### connections.json
Maps service names to Scalekit connection names:

```json
{
  "gmail": {
    "connection_name": "gmail_u3134a",
    "identifier": "mess"
  },
  "slack": {
    "connection_name": "slack_x7y9z",
    "identifier": "mess"
  }
}
```

**Note:** `identifier` is auto-set to agent's name (from IDENTITY.md).

### .env
Scalekit API credentials (never commit to git!):

```bash
SCALEKIT_CLIENT_ID=sk_live_...
SCALEKIT_CLIENT_SECRET=...
SCALEKIT_ENV_URL=https://...
```

## Supported Services

Any OAuth provider Scalekit supports:
- Gmail, Google Calendar, Google Drive
- Slack, Notion, Linear, GitHub
- Salesforce, HubSpot, Zendesk
- 50+ more

Check [Scalekit Connectors](https://docs.scalekit.com/connectors) for full list.

## Authorization Flow

```
1. Agent calls get_token("gmail")
2. Check if connection configured → if NO, ask user
3. Check if authorized (status == ACTIVE)
4. If NOT authorized:
   - Generate auth link (expires 1 min)
   - Send to user via Telegram/chat
   - Wait for authorization
5. Return fresh access_token
6. Scalekit auto-refreshes in background
```

## Error Handling

**Connection not configured:**
```
Error: gmail not configured. Please:
1. Create connection in Scalekit dashboard
2. Provide connection_name
```

**Authorization expired:**
```
Authorization needed: [link]
(Link expires in 1 minute - click now!)
```

**Scalekit credentials missing:**
```
Scalekit not configured. Please provide:
- SCALEKIT_CLIENT_ID
- SCALEKIT_CLIENT_SECRET
- SCALEKIT_ENV_URL
```

## Security Best Practices

1. **Never log tokens** - use `[REDACTED]` in logs
2. **Add .env to .gitignore** - never commit credentials
3. **Rotate credentials** if exposed
4. **Use separate Scalekit accounts** for dev/prod
5. **Auth links expire in 1 min** - act fast!

## Troubleshooting

**"Module not found" error:**
```bash
cd skills/scalekit-auth
pip3 install -r requirements.txt
```

**Token returns 401:**
- Authorization may have expired
- Agent will prompt for re-authorization

**Connection not found:**
- Check `connections.json` exists
- Verify connection_name from Scalekit dashboard

## Example: Gmail Integration

```python
# In your skill's script
from scalekit_helper import get_token
import requests

def fetch_unread_emails():
    token = get_token("gmail")
    
    headers = {"Authorization": f"Bearer {token}"}
    url = "https://gmail.googleapis.com/gmail/v1/users/me/messages"
    params = {"q": "is:unread", "maxResults": 5}
    
    response = requests.get(url, headers=headers, params=params)
    return response.json()
```

## Publishing Skills with Scalekit Auth

If your skill uses scalekit-auth:

1. **Document in SKILL.md:**
   ```markdown
   ## Prerequisites
   - Install scalekit-auth skill
   - Configure [SERVICE] connection in Scalekit
   ```

2. **Import in scripts:**
   ```python
   sys.path.append('./skills/scalekit-auth')
   from scalekit_helper import get_token
   ```

3. **Handle errors gracefully** - guide users to configure connections

## API Reference

### get_token(service_name: str) → str

Returns fresh OAuth access token for the service.

**Parameters:**
- `service_name`: Service identifier (e.g., "gmail", "slack")

**Returns:**
- `access_token`: Fresh OAuth bearer token

**Raises:**
- `ConfigurationError`: Service not configured or Scalekit creds missing
- `AuthorizationError`: User needs to authorize (sends link to user)

**Example:**
```python
token = get_token("gmail")
print(f"Token: {token[:10]}...")  # Never log full token!
```

## Roadmap

- [ ] Multi-user support (multiple identifiers per service)
- [ ] Token caching (reduce API calls)
- [ ] CLI tool (`scalekit-auth config gmail gmail_u3134a`)
- [ ] Auto-detect service from API URL
- [ ] Batch token retrieval

## Contributing

Found a bug? Have a feature request? Open an issue on ClawHub!

---

**Remember:** Tokens are secrets. Handle with care. 🔐
