#!/usr/bin/env bash
set -euo pipefail

# SatGate CLI Configuration
# Interactive setup for ~/.satgate/config.yaml

CONFIG_DIR="${HOME}/.satgate"
CONFIG_FILE="${CONFIG_DIR}/config.yaml"

echo "⚡ SatGate CLI Configuration"
echo "───────────────────────────"
echo ""

# Check if config already exists
if [ -f "$CONFIG_FILE" ]; then
  echo "  Existing config found at ${CONFIG_FILE}"
  read -p "  Overwrite? [y/N]: " OVERWRITE
  if [ "${OVERWRITE,,}" != "y" ]; then
    echo "  Keeping existing config."
    exit 0
  fi
fi

# Choose surface
echo ""
echo "  Where is your SatGate gateway?"
echo "    1) Self-hosted (local or remote gateway)"
echo "    2) SatGate Cloud (cloud.satgate.io)"
echo ""
read -p "  Choice [1/2]: " SURFACE_CHOICE

if [ "$SURFACE_CHOICE" = "2" ]; then
  SURFACE="cloud"
  GATEWAY="https://cloud.satgate.io"

  echo ""
  echo "  SatGate Cloud requires a session token."
  echo "  Get one by signing in at https://cloud.satgate.io"
  echo "  or via magic link: satgate auth login (coming soon)"
  echo ""
  read -p "  Session token: " SESSION_TOKEN

  read -p "  Tenant slug (press Enter for default): " TENANT
  TENANT="${TENANT:-default}"

  # Write config
  mkdir -p "$CONFIG_DIR"
  cat > "$CONFIG_FILE" <<EOF
# SatGate CLI Configuration
# Generated by configure.sh

surface: cloud
gateway: ${GATEWAY}
session_token: ${SESSION_TOKEN}
tenant: ${TENANT}
format: table
EOF

else
  SURFACE="gateway"

  echo ""
  read -p "  Gateway URL [http://localhost:9090]: " GATEWAY
  GATEWAY="${GATEWAY:-http://localhost:9090}"

  read -p "  Admin token: " ADMIN_TOKEN

  # Write config
  mkdir -p "$CONFIG_DIR"
  cat > "$CONFIG_FILE" <<EOF
# SatGate CLI Configuration
# Generated by configure.sh

surface: gateway
gateway: ${GATEWAY}
admin_token: ${ADMIN_TOKEN}
format: table
EOF

fi

chmod 600 "$CONFIG_FILE"

echo ""
echo "✓ Config written to ${CONFIG_FILE}"
echo ""

# Test connection
echo "  Testing connection..."
if satgate ping 2>/dev/null; then
  echo ""
  echo "  🎉 You're connected! Try: satgate status"
else
  echo ""
  echo "  ⚠️  Could not reach gateway. Check your config:"
  echo "     ${CONFIG_FILE}"
fi
