#!/bin/bash
set -e

echo "🗑️  Uninstalling Rune - Persistent AI Memory System..."

# Ask for confirmation
read -p "⚠️  This will remove all memory data. Are you sure? (y/N) " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "❌ Uninstall cancelled"
    exit 1
fi

# Backup memory before deletion (optional)
MEMORY_DB="$HOME/.openclaw/memory.db"
if [[ -f "$MEMORY_DB" ]]; then
    BACKUP_FILE="$HOME/.openclaw/memory_backup_$(date +%Y%m%d_%H%M%S).db"
    echo "💾 Creating backup at $BACKUP_FILE"
    cp "$MEMORY_DB" "$BACKUP_FILE"
fi

# Remove global CLI
echo "🔧 Removing brokkr-mem CLI..."
npm uninstall -g brokkr-mem 2>/dev/null || echo "   (CLI not globally installed)"

# Remove memory database
if [[ -f "$MEMORY_DB" ]]; then
    echo "🗄️ Removing memory database..."
    rm "$MEMORY_DB"
fi

# Remove FACTS.md if it was generated by Rune
FACTS_FILE="$HOME/.openclaw/workspace/FACTS.md"
if [[ -f "$FACTS_FILE" ]] && grep -q "Generated by brokkr-mem" "$FACTS_FILE" 2>/dev/null; then
    echo "📋 Removing generated FACTS.md..."
    rm "$FACTS_FILE"
fi

# Clean up heartbeat integration
HEARTBEAT_FILE="$HOME/.openclaw/workspace/HEARTBEAT.md"
if [[ -f "$HEARTBEAT_FILE" ]] && grep -q "brokkr-mem" "$HEARTBEAT_FILE"; then
    echo "📝 Cleaning HEARTBEAT.md integration..."
    # Remove Rune section (basic cleanup)
    sed -i '/## 🧠 Rune Memory Maintenance/,/^$/d' "$HEARTBEAT_FILE" 2>/dev/null || true
    sed -i '/brokkr-mem/d' "$HEARTBEAT_FILE" 2>/dev/null || true
fi

# Remove project directories  
if [[ -d "node_modules" ]]; then
    echo "📦 Removing dependencies..."
    rm -rf node_modules
fi

echo ""
echo "✅ Rune uninstall complete!"
echo ""
echo "💾 Memory backup saved at: $BACKUP_FILE"
echo "🔄 Restart your OpenClaw session to complete removal"
echo ""
echo "😢 Your AI will no longer have persistent memory."
echo "   Reinstall anytime with: clawhub install rune"