#!/usr/bin/env python3
"""
format_digest.py — 将 JSON 格式的文章数据格式化为 Markdown 日报
"""
import argparse
import json
from datetime import datetime, timezone
from collections import defaultdict




EMOJI_MAP = {
    "Tech": "🔧", "Business": "💼", "World": "🌍",
    "AI": "🤖", "Science": "🔬", "Other": "📌"
}




def format_digest(articles_data, template_path=None):
    """将文章按分类整理成 Markdown 格式日报"""
    articles = articles_data.get("articles", [])
    errors = articles_data.get("errors", [])
    now = datetime.now(timezone.utc)


    # 按分类分组
    by_category = defaultdict(list)
    for a in articles:
        by_category[a.get("category", "Other")].append(a)


    # 构建 Markdown
    lines = []
    lines.append(
        f"# Daily Digest — {now.strftime('%Y-%m-%d')}\n"
    )
    lines.append(
        f"> **{len(articles)}** articles from "
        f"**{articles_data.get('sources_checked', '?')}** sources\n"
    )


    for category, items in sorted(by_category.items()):
        emoji = EMOJI_MAP.get(category, "📌")
        lines.append(f"\n## {emoji} {category}\n")
        for item in items[:15]:  # 每分类最多 15 篇
            title = item.get("title", "Untitled")
            url = item.get("url", "#")
            source = item.get("source", "Unknown")
            desc = item.get(
                "ai_summary", item.get("description", "")[:80]
            )
            lines.append(
                f"- **[{title}]({url})** — {desc} *({source})*"
            )


    # 错误报告
    if errors:
        lines.append(f"\n## Feed Errors ({len(errors)})\n")
        for err in errors:
            lines.append(f"- {err['source']}: {err['error']}")


    lines.append(
        f"\n---\n*Generated by RSS Daily Digest Skill "
        f"at {now.strftime('%Y-%m-%d %H:%M UTC')}*"
    )
    return "\n".join(lines)




def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--input', required=True, help='JSON articles file')
    parser.add_argument('--template', help='Optional template path')
    args = parser.parse_args()


    with open(args.input, 'r', encoding='utf-8') as f:
        data = json.load(f)


    digest = format_digest(data, args.template)
    print(digest)




if __name__ == "__main__":
    main()
