---
name: rose-docker-build
description: Build the ROSE compiler in a Docker container using autotools or CMake. Use when setting up ROSE development environment, building ROSE from source, or troubleshooting ROSE build issues. ROSE requires GCC 7-10 which most modern hosts don't have, so Docker is the recommended approach.
---

# ROSE Docker Build

Build the ROSE source-to-source compiler in an isolated Docker container.

## Why Docker?

ROSE requires GCC 7-10. Modern systems have GCC 11+, causing build failures. Docker provides:
- Correct GCC version (9.x recommended)
- All dependencies pre-installed
- Reproducible builds
- Edit on host, build in container

## Quick Start (Autotools)

```bash
# 1. Clone ROSE
git clone git@github.com:rose-compiler/rose.git
cd rose && git checkout weekly

# 2. Create Docker environment
mkdir ../rose-docker && cd ../rose-docker

# 3. Build and run container (see Dockerfile below)
docker build -t rose-dev .
docker run -d --name rose-dev -v $(pwd)/../rose:/rose/src rose-dev

# 4. Build ROSE inside container
docker exec rose-dev bash -c 'cd /rose/src && ./build'
docker exec rose-dev bash -c 'mkdir -p /rose/build && cd /rose/build && \
  /rose/src/configure --prefix=/rose/install \
    --enable-languages=c,c++ \
    --with-boost=/usr \
    --with-boost-libdir=/usr/lib/x86_64-linux-gnu \
    --disable-binary-analysis \
    --disable-java'
docker exec rose-dev bash -c 'cd /rose/build && make core -j$(nproc)'
docker exec rose-dev bash -c 'cd /rose/build && make install-core'
```

## Quick Start (CMake)

CMake builds require CMake 4.x (3.16 fails at ROSETTA generation).

```bash
# 1. Clone ROSE
git clone git@github.com:rose-compiler/rose.git
cd rose && git checkout weekly

# 2. Create Docker environment and build container
mkdir ../rose-docker && cd ../rose-docker
docker build -t rose-dev -f Dockerfile.cmake .
docker run -d --name rose-cmake -v $(pwd)/../rose:/rose/src:ro rose-dev

# 3. Configure with CMake
docker exec -w /rose/build rose-cmake cmake /rose/src \
  -DCMAKE_INSTALL_PREFIX=/rose/install \
  -DENABLE_C=ON \
  -DENABLE_TESTS=OFF \
  -DCMAKE_BUILD_TYPE=Release

# 4. Build (use -j4 to avoid OOM on 16GB systems)
docker exec -w /rose/build rose-cmake make -j4

# 5. Test
docker exec rose-cmake /rose/build/bin/rose-compiler --version
```

### CMake Options

| Option | Description |
|--------|-------------|
| `ENABLE_C=ON` | Enable C/C++ analysis (uses EDG frontend) |
| `ENABLE_BINARY_ANALYSIS=ON` | Enable binary analysis (no EDG needed) |
| `ENABLE_TESTS=OFF` | Skip test compilation (faster build) |
| `CMAKE_BUILD_TYPE=Release` | Optimized build |

### CMake vs Autotools

| Feature | Autotools | CMake |
|---------|-----------|-------|
| Stability | ✅ Mature | ⚠️ Newer |
| C/C++ analysis | ✅ Works | ✅ Works (with fixes) |
| Build target | `make core` | `make` (full build) |
| Incremental builds | Slower | Faster |
| IDE integration | Limited | Excellent |

## Dockerfiles

### Dockerfile (Autotools)

```dockerfile
FROM ubuntu:20.04
ENV DEBIAN_FRONTEND=noninteractive
ENV TZ=America/Los_Angeles

RUN apt-get update && apt-get install -y \
    build-essential g++ gcc gfortran \
    automake autoconf libtool flex bison \
    libboost-all-dev libxml2-dev \
    git wget curl vim \
    && rm -rf /var/lib/apt/lists/*

RUN useradd -m -s /bin/bash developer
RUN mkdir -p /rose/src /rose/build /rose/install && chown -R developer:developer /rose
USER developer
WORKDIR /rose
CMD ["tail", "-f", "/dev/null"]
```

### Dockerfile.cmake (CMake)

```dockerfile
FROM ubuntu:20.04
ENV DEBIAN_FRONTEND=noninteractive
ENV TZ=America/Los_Angeles

# Install build dependencies
RUN apt-get update && apt-get install -y \
    build-essential g++ gcc gfortran \
    flex bison \
    libboost-all-dev libxml2-dev libreadline-dev \
    zlib1g-dev libsqlite3-dev libpq-dev libyaml-dev \
    libgmp-dev libmpc-dev libmpfr-dev \
    git wget curl vim \
    gnupg software-properties-common \
    && rm -rf /var/lib/apt/lists/*

# Install CMake 4.x from Kitware (Ubuntu 20.04 has 3.16 which is too old)
RUN wget -O - https://apt.kitware.com/keys/kitware-archive-latest.asc 2>/dev/null | apt-key add - \
    && echo 'deb https://apt.kitware.com/ubuntu/ focal main' > /etc/apt/sources.list.d/kitware.list \
    && apt-get update && apt-get install -y cmake \
    && rm -rf /var/lib/apt/lists/*

RUN useradd -m -s /bin/bash developer
RUN mkdir -p /rose/src /rose/build /rose/install && chown -R developer:developer /rose
USER developer
WORKDIR /rose
CMD ["tail", "-f", "/dev/null"]
```

## EDG Binary Handling

EDG (C++ frontend) binaries are required for C/C++ analysis. The build checks:

### Autotools
1. Build directory for existing tarball
2. Source tree at `src/frontend/CxxFrontend/roseBinaryEDG-*.tar.gz`
3. Network download from `edg-binaries.rosecompiler.org`

### CMake
1. Source tree at `src/frontend/CxxFrontend/roseBinaryEDG-*.tar.gz` (auto-detected)
2. Extracts matching tarball based on GCC version at build time
3. Links `libroseEDG.a` to `librose.so` automatically

If server is down, ensure source tree has EDG binaries (included in `weekly` branch).

## Common Commands

### Autotools
```bash
# Rebuild after source changes
docker exec rose-dev bash -c 'cd /rose/build && make core -j8'

# Install
docker exec rose-dev bash -c 'cd /rose/build && make install-core'
```

### CMake
```bash
# Rebuild after source changes  
docker exec -w /rose/build rose-cmake make -j4

# Install
docker exec -w /rose/build rose-cmake make install
```

### Both
```bash
# Test ROSE compiler
docker exec rose-dev /rose/install/bin/rose-compiler --version
docker exec rose-cmake /rose/build/bin/rose-compiler --version

# Source-to-source test
docker exec rose-dev bash -c 'echo "int main(){return 0;}" > /tmp/test.c && \
  /rose/install/bin/rose-compiler -c /tmp/test.c && cat rose_test.c'

# Enter container shell
docker exec -it rose-dev bash
docker exec -it rose-cmake bash
```

## Build Time

| Build System | First Build | Incremental |
|--------------|-------------|-------------|
| Autotools (`make core -j8`) | ~60-90 min | seconds-minutes |
| CMake (`make -j4`) | ~35 min | seconds-minutes |

- `librose.so`: ~200 MB (CMake) to ~1.3 GB (autotools with debug)
- Memory: Use `-j4` on 16GB systems to avoid OOM

## Troubleshooting

| Issue | Solution |
|-------|----------|
| EDG download fails | Use `weekly` branch (has EDG binaries in source tree) |
| CMake: ROSETTA fails | Upgrade to CMake 4.x |
| CMake: EDG link errors | Ensure using latest CMake fixes (PR #250) |
| CMake: quadmath errors | Add `-lquadmath` or use latest fixes |
| Permission denied | Check volume mount permissions |
| Out of memory | Reduce `-j` parallelism |
| Boost not found | Verify boost paths in configure/cmake |

## Testing with Sample Code

```bash
# Create factorial test
cat << 'EOF' | docker exec -i rose-cmake tee /tmp/factorial.cpp
#include <iostream>
int factorial(int n) { return n <= 1 ? 1 : n * factorial(n-1); }
int main() {
    for(int i = 0; i <= 10; i++)
        std::cout << "factorial(" << i << ") = " << factorial(i) << std::endl;
    return 0;
}
EOF

# Run through ROSE (source-to-source transformation)
docker exec -w /tmp rose-cmake /rose/build/bin/rose-compiler factorial.cpp

# Compile and run the transformed code
docker exec -w /tmp rose-cmake g++ rose_factorial.cpp -o factorial_test
docker exec -w /tmp rose-cmake ./factorial_test
```

Expected output:
```
factorial(0) = 1
factorial(1) = 1
...
factorial(10) = 3628800
```
