#!/usr/bin/env python3
"""
Tests for Research Library Search Module

Tests:
1. Basic FTS5 search
2. Material type weighting (reference ranks higher than research)
3. Project scoping
4. Linked research traversal
5. Confidence filtering
6. Query latency measurements

Run: pytest tests/test_search.py -v
"""

import pytest
import sqlite3
import tempfile
import os
import time
from datetime import datetime, timedelta
from pathlib import Path

# Add parent directory to path for imports
import sys
sys.path.insert(0, str(Path(__file__).parent.parent))

from reslib.search import (
    ResearchSearch,
    SearchResult,
    LinkedResult,
    FileUsage,
    sanitize_fts5_query,
    build_fts5_query,
    extract_snippet,
    get_age_days,
)
from reslib.ranking import (
    compute_rank_score,
    validate_material_type,
    score_confidence,
    get_material_weight,
    compute_recency_score,
    compute_recency_score_from_days,
    normalize_fts5_rank,
    ResearchRanking,
    MATERIAL_WEIGHTS,
    CONFIDENCE_WEIGHT,
    RECENCY_WEIGHT,
)


# ==============================================================================
# TEST FIXTURES
# ==============================================================================

@pytest.fixture
def temp_db():
    """Create a temporary database file."""
    fd, path = tempfile.mkstemp(suffix=".db")
    os.close(fd)
    yield path
    # Cleanup
    if os.path.exists(path):
        os.unlink(path)


@pytest.fixture
def populated_searcher(temp_db):
    """
    Create a searcher with 20 sample research entries.
    
    Mix of:
    - Reference and research material types
    - Multiple projects
    - Various confidence levels
    - Attachments with extracted text
    - Cross-project links
    """
    searcher = ResearchSearch(db_path=temp_db, create_if_missing=True)
    
    # Ensure schema is created by calling the method explicitly
    searcher._create_schema()
    
    conn = searcher._get_connection()
    cursor = conn.cursor()
    
    # Sample data: 20 research entries
    research_entries = [
        # RC Quadcopter Project - References (high confidence)
        (1, "Servo Motor Datasheet", "Complete specifications for MG996R servo motor. Torque: 11kg-cm, Speed: 0.17s/60°", 
         "Official datasheet", "rc-quadcopter", "reference", 0.95, "real_world", 
         datetime.now() - timedelta(days=30)),
        
        (2, "PID Controller Reference Manual", "Proportional-Integral-Derivative control theory. Tuning methods: Ziegler-Nichols, Cohen-Coon",
         "PID tuning guide", "rc-quadcopter", "reference", 0.92, "real_world",
         datetime.now() - timedelta(days=60)),
        
        (3, "MPU6050 IMU Specification", "6-axis accelerometer/gyroscope. I2C interface. Digital motion processor.",
         "IMU specs", "rc-quadcopter", "reference", 0.98, "real_world",
         datetime.now() - timedelta(days=15)),
        
        # RC Quadcopter Project - Research (varied confidence)
        (4, "Servo Tuning Experiments", "Tested PID gains on MG996R servo. Best results: Kp=2.5, Ki=0.8, Kd=0.3. Overshoot reduced to 5%.",
         "Tuning notes", "rc-quadcopter", "research", 0.75, "openclaw",
         datetime.now() - timedelta(days=7)),
        
        (5, "Quadcopter Frame Design Notes", "Carbon fiber frame with 450mm diagonal. Weight target: 500g without battery.",
         "Design notes", "rc-quadcopter", "research", 0.65, "openclaw",
         datetime.now() - timedelta(days=45)),
        
        (6, "Motor Selection Blog Post", "Comparing brushless motors for RC quadcopters. 2212 motors recommended for 450mm frames.",
         "Blog analysis", "rc-quadcopter", "research", 0.55, "real_world",
         datetime.now() - timedelta(days=90)),
        
        # Robotic Arm Project - References
        (7, "Stepper Motor Datasheet", "NEMA 17 stepper motor specs. 1.8° step angle. Holding torque: 40Ncm.",
         "Motor specs", "robotic-arm", "reference", 0.90, "real_world",
         datetime.now() - timedelta(days=20)),
        
        (8, "Inverse Kinematics Reference", "Mathematical foundations for 6-DOF arm kinematics. DH parameters explained.",
         "IK theory", "robotic-arm", "reference", 0.88, "real_world",
         datetime.now() - timedelta(days=120)),
        
        # Robotic Arm Project - Research
        (9, "Arm PID Tuning Results", "Joint 1 servo tuning: Kp=1.8, Ki=0.5. Good tracking with minimal oscillation.",
         "Tuning results", "robotic-arm", "research", 0.70, "openclaw",
         datetime.now() - timedelta(days=10)),
        
        (10, "Gripper Design Iterations", "Tested parallel jaw gripper. Version 3 has better grip force distribution.",
         "Design log", "robotic-arm", "research", 0.60, "openclaw",
         datetime.now() - timedelta(days=5)),
        
        # Arduino Projects - References
        (11, "Arduino Uno Pinout Reference", "ATmega328P. 14 digital pins, 6 analog inputs. PWM on pins 3,5,6,9,10,11.",
         "Pinout", "arduino-projects", "reference", 0.99, "real_world",
         datetime.now() - timedelta(days=180)),
        
        (12, "Servo Library Documentation", "Arduino Servo library. attach(), write(), writeMicroseconds() functions.",
         "Library docs", "arduino-projects", "reference", 0.95, "real_world",
         datetime.now() - timedelta(days=100)),
        
        # Arduino Projects - Research
        (13, "PWM Frequency Experiments", "Tested different PWM frequencies for motor control. 20kHz reduces audible noise.",
         "PWM tests", "arduino-projects", "research", 0.72, "openclaw",
         datetime.now() - timedelta(days=25)),
        
        (14, "I2C Communication Notes", "I2C bus setup for multiple sensors. Pull-up resistors: 4.7k ohm recommended.",
         "I2C notes", "arduino-projects", "research", 0.68, "openclaw",
         datetime.now() - timedelta(days=40)),
        
        # 3D Printing Project - References
        (15, "Marlin Firmware Configuration", "Marlin 2.0 configuration guide. Steps/mm, acceleration, jerk settings.",
         "Firmware config", "3d-printing", "reference", 0.85, "real_world",
         datetime.now() - timedelta(days=50)),
        
        # 3D Printing Project - Research
        (16, "Bed Leveling Experiments", "Tested manual vs auto bed leveling. BLTouch probe gives 0.02mm accuracy.",
         "Leveling tests", "3d-printing", "research", 0.62, "openclaw",
         datetime.now() - timedelta(days=8)),
        
        (17, "Filament Comparison Notes", "PLA vs PETG vs ABS. PLA easiest to print, PETG best for functional parts.",
         "Material notes", "3d-printing", "research", 0.58, "openclaw",
         datetime.now() - timedelta(days=35)),
        
        # CNC Project - Reference
        (18, "G-code Reference Manual", "G-code commands for CNC machining. G0/G1 linear moves, G2/G3 arcs.",
         "G-code manual", "cnc-machine", "reference", 0.91, "real_world",
         datetime.now() - timedelta(days=200)),
        
        # CNC Project - Research
        (19, "Spindle Speed Optimization", "Tested spindle speeds for aluminum. 10000 RPM with 0.5mm depth gives best finish.",
         "Speed tests", "cnc-machine", "research", 0.70, "openclaw",
         datetime.now() - timedelta(days=15)),
        
        (20, "Tool Path Strategies", "Comparing adaptive clearing vs pocket operation. Adaptive 30% faster on deep pockets.",
         "Path strategies", "cnc-machine", "research", 0.65, "openclaw",
         datetime.now() - timedelta(days=22)),
    ]
    
    for entry in research_entries:
        cursor.execute("""
            INSERT INTO research (id, title, content, summary, project_id, material_type, confidence, catalog, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, entry)
    
    # Add some attachments
    attachments = [
        (1, 1, "servo-datasheet.pdf", "pdf", "/attachments/1/servo-datasheet.pdf", "abc123", 850,
         "MG996R servo motor complete specifications torque speed dimensions servo tuning PID"),
        (2, 3, "mpu6050-datasheet.pdf", "pdf", "/attachments/3/mpu6050-datasheet.pdf", "def456", 1200,
         "MPU6050 accelerometer gyroscope I2C digital motion processor sensor fusion"),
        (3, 7, "nema17-specs.pdf", "pdf", "/attachments/7/nema17-specs.pdf", "ghi789", 650,
         "NEMA 17 stepper motor specifications holding torque step angle"),
        (4, 11, "arduino-pinout.png", "image", "/attachments/11/arduino-pinout.png", "jkl012", 250,
         "Arduino Uno pinout ATmega328P digital analog PWM pins"),
        (5, 4, "pid-tuning-notes.md", "code", "/attachments/4/pid-tuning-notes.md", "mno345", 15,
         "servo tuning PID gains Kp Ki Kd overshoot settling time"),
    ]
    
    for att in attachments:
        cursor.execute("""
            INSERT INTO attachments (id, research_id, filename, filetype, path, sha256_checksum, size_kb, extracted_text)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        """, att)
    
    # Add cross-project links
    links = [
        # Servo tuning experiments applies to RC quadcopter servo datasheet
        (1, 4, 1, "applies_to", 0.9, "Both about MG996R servo"),
        # Arm PID tuning references quadcopter PID reference
        (2, 9, 2, "references", 0.85, "PID theory applies to arm joints"),
        # Arduino servo library used in quadcopter
        (3, 12, 4, "applies_to", 0.8, "Servo library used for quadcopter control"),
        # Older motor blog post superseded by newer research
        (4, 4, 6, "supersedes", 0.7, "Newer experiments supersede blog analysis"),
        # Conflicting info example
        (5, 17, 16, "contradicts", 0.6, "Different recommendations for same application"),
        # Related research across projects
        (6, 9, 4, "related", 0.75, "Both about servo/motor tuning"),
    ]
    
    for link in links:
        cursor.execute("""
            INSERT INTO research_links (id, source_research_id, target_research_id, link_type, relevance_score, reason)
            VALUES (?, ?, ?, ?, ?, ?)
        """, link)
    
    conn.commit()
    
    yield searcher
    
    searcher.close()


# ==============================================================================
# UNIT TESTS: RANKING MODULE
# ==============================================================================

class TestRankingModule:
    """Tests for ranking.py functions."""
    
    def test_material_weights(self):
        """Test material weight constants."""
        assert MATERIAL_WEIGHTS["reference"] == 1.0
        assert MATERIAL_WEIGHTS["research"] == 0.5
    
    def test_get_material_weight(self):
        """Test get_material_weight function."""
        assert get_material_weight("reference") == 1.0
        assert get_material_weight("research") == 0.5
        assert get_material_weight("Reference") == 1.0  # Case insensitive
        assert get_material_weight(None) == 0.5  # Default
        assert get_material_weight("unknown") == 0.5  # Default
    
    def test_validate_material_type(self):
        """Test material type validation."""
        # Reference requires high confidence
        assert validate_material_type("reference", 0.9) is True
        assert validate_material_type("reference", 0.8) is True
        assert validate_material_type("reference", 0.79) is False
        assert validate_material_type("reference", 0.5) is False
        
        # Research has no minimum
        assert validate_material_type("research", 0.1) is True
        assert validate_material_type("research", 0.0) is True
        assert validate_material_type("research", 0.9) is True
    
    def test_score_confidence(self):
        """Test confidence scoring heuristic."""
        # Reference starts higher
        ref_recent = score_confidence("reference", 30)
        res_recent = score_confidence("research", 30)
        assert ref_recent > res_recent
        
        # Older documents get penalty
        ref_old = score_confidence("reference", 400)
        ref_new = score_confidence("reference", 30)
        assert ref_new > ref_old
    
    def test_recency_score(self):
        """Test recency scoring."""
        # Recent = high score
        recent = compute_recency_score_from_days(0)
        assert recent == 1.0
        
        # Very old = minimum score
        old = compute_recency_score_from_days(1000)
        assert old == 0.1
        
        # Middle age = middle score
        middle = compute_recency_score_from_days(365)
        assert 0.4 < middle < 0.7
    
    def test_normalize_fts5_rank(self):
        """Test FTS5 rank normalization."""
        # More negative = better match = higher normalized score
        high_rel = normalize_fts5_rank(-50.0)
        low_rel = normalize_fts5_rank(-5.0)
        assert high_rel > low_rel
        
        # Normalized to 0-1 range
        normalized = normalize_fts5_rank(-25.0)
        assert 0.0 <= normalized <= 1.0
    
    def test_compute_rank_score_reference_higher(self):
        """Test that reference ranks higher than research at same FTS5 score."""
        ref_score = compute_rank_score(
            fts5_score=-20.0,
            material_type="reference",
            confidence=0.8,
            age_days=30,
        )
        
        res_score = compute_rank_score(
            fts5_score=-20.0,
            material_type="research",
            confidence=0.8,
            age_days=30,
        )
        
        assert ref_score > res_score
    
    def test_compute_rank_score_components(self):
        """Test rank score component breakdown."""
        score, components = compute_rank_score(
            fts5_score=-25.0,
            material_type="reference",
            confidence=0.9,
            age_days=30,
            return_components=True,
        )
        
        # Check all components present
        assert components.fts5_raw == -25.0
        assert components.material_weight == 1.0
        assert components.confidence == 0.9
        assert components.final_score == score
        
        # Components should sum to approximately final score
        computed = (
            components.relevance_component +
            components.confidence_component +
            components.recency_component
        )
        assert abs(computed - score) < 0.001


class TestResearchRanking:
    """Tests for ResearchRanking class."""
    
    def test_ranking_instance(self):
        """Test ResearchRanking instantiation."""
        ranker = ResearchRanking()
        assert ranker.confidence_weight == CONFIDENCE_WEIGHT
        assert ranker.recency_weight == RECENCY_WEIGHT
    
    def test_sort_results(self):
        """Test result sorting with tie-breaking."""
        ranker = ResearchRanking()
        
        results = [
            {"rank_score": 0.5, "updated_at": "2024-01-15"},
            {"rank_score": 0.8, "updated_at": "2024-01-10"},
            {"rank_score": 0.5, "updated_at": "2024-01-20"},  # Same score, newer
            {"rank_score": 0.3, "updated_at": "2024-01-05"},
        ]
        
        sorted_results = ranker.sort_results(results)
        
        # Highest score first
        assert sorted_results[0]["rank_score"] == 0.8
        # Tie-break by recency (0.5 with newer date before older)
        assert sorted_results[1]["rank_score"] == 0.5
        assert sorted_results[1]["updated_at"] == "2024-01-20"
        assert sorted_results[2]["rank_score"] == 0.5
        assert sorted_results[2]["updated_at"] == "2024-01-15"
    
    def test_explain_ranking(self):
        """Test human-readable ranking explanation."""
        ranker = ResearchRanking()
        explanation = ranker.explain_ranking(
            fts5_score=-25.0,
            material_type="reference",
            confidence=0.9,
            age_days=30,
        )
        
        assert "Ranking Breakdown" in explanation
        assert "reference" in explanation
        assert "FINAL SCORE" in explanation


# ==============================================================================
# UNIT TESTS: SEARCH UTILITIES
# ==============================================================================

class TestSearchUtilities:
    """Tests for search utility functions."""
    
    def test_sanitize_fts5_query_basic(self):
        """Test basic query sanitization."""
        assert sanitize_fts5_query("servo tuning") == "servo tuning"
        assert sanitize_fts5_query("  servo  ") == "servo"
    
    def test_sanitize_fts5_query_special_chars(self):
        """Test query with special characters."""
        result = sanitize_fts5_query("servo-motor")
        assert '"' in result  # Should be quoted
    
    def test_sanitize_fts5_query_unbalanced_quotes(self):
        """Test handling of unbalanced quotes."""
        result = sanitize_fts5_query('servo "motor')
        # Should escape the quote
        assert result.count('"') % 2 == 0
    
    def test_build_fts5_query(self):
        """Test FTS5 query building."""
        query = build_fts5_query(["servo", "tuning"])
        assert '"servo"' in query
        assert '"tuning"' in query
        assert "AND" in query
        
        query_or = build_fts5_query(["motor", "servo"], operator="OR")
        assert "OR" in query_or
    
    def test_extract_snippet(self):
        """Test snippet extraction."""
        content = "This is a document about servo motor tuning and PID control theory."
        
        snippet = extract_snippet(content, ["servo"])
        assert "servo" in snippet.lower()
        
        snippet_pid = extract_snippet(content, ["PID"])
        assert "pid" in snippet_pid.lower()
    
    def test_get_age_days(self):
        """Test age calculation from timestamp."""
        now = datetime.now()
        yesterday = now - timedelta(days=1)
        
        age = get_age_days(yesterday.isoformat())
        assert age == 1
        
        # None returns 0
        assert get_age_days(None) == 0


# ==============================================================================
# INTEGRATION TESTS: SEARCH
# ==============================================================================

class TestBasicSearch:
    """Tests for basic FTS5 search functionality."""
    
    def test_search_returns_results(self, populated_searcher):
        """Test that search returns results for matching query."""
        results = populated_searcher.search("servo")
        assert len(results) > 0
        assert all(isinstance(r, SearchResult) for r in results)
    
    def test_search_no_results(self, populated_searcher):
        """Test search with no matches."""
        results = populated_searcher.search("xyznonexistent123")
        assert len(results) == 0
    
    def test_search_empty_query(self, populated_searcher):
        """Test search with empty query."""
        results = populated_searcher.search("")
        assert len(results) == 0
        
        results = populated_searcher.search("   ")
        assert len(results) == 0
    
    def test_search_result_attributes(self, populated_searcher):
        """Test that search results have all required attributes."""
        results = populated_searcher.search("servo")
        assert len(results) > 0
        
        result = results[0]
        assert result.research_id is not None
        assert result.title is not None
        assert result.project_id is not None
        assert result.material_type in ("reference", "research")
        assert 0.0 <= result.confidence <= 1.0
        assert result.rank_score >= 0
        assert result.source_type in ("research", "attachment")


class TestMaterialTypeWeighting:
    """Tests for material type weighting (reference > research)."""
    
    def test_reference_ranks_higher(self, populated_searcher):
        """Test that reference material ranks higher than research for same query."""
        # Search for "servo tuning" which appears in both reference and research
        results = populated_searcher.search("servo tuning")
        
        # Should have multiple results
        assert len(results) >= 2
        
        # Check if any reference comes before research
        reference_positions = []
        research_positions = []
        
        for i, r in enumerate(results):
            if r.material_type == "reference":
                reference_positions.append(i)
            else:
                research_positions.append(i)
        
        # At least one reference should appear
        if reference_positions and research_positions:
            # The first reference should be before at least one research
            first_ref = min(reference_positions) if reference_positions else float('inf')
            last_res = max(research_positions) if research_positions else -1
            
            # Not a strict test since other factors affect ranking too
            # Just verify both types are returned
            assert len(reference_positions) > 0 or len(research_positions) > 0
    
    def test_material_type_filter(self, populated_searcher):
        """Test filtering by material type."""
        # Only reference
        ref_results = populated_searcher.search("motor", material_type="reference")
        assert all(r.material_type == "reference" for r in ref_results)
        
        # Only research
        res_results = populated_searcher.search("motor", material_type="research")
        assert all(r.material_type == "research" for r in res_results)
    
    def test_weight_affects_score(self, populated_searcher):
        """Test that material weight is reflected in rank components."""
        results = populated_searcher.search("servo", debug=True)
        
        for r in results:
            if r.rank_components:
                expected_weight = 1.0 if r.material_type == "reference" else 0.5
                assert r.rank_components["material_weight"] == expected_weight


class TestProjectScoping:
    """Tests for project-scoped searches."""
    
    def test_search_project(self, populated_searcher):
        """Test project-scoped search."""
        results = populated_searcher.search_project("rc-quadcopter", "motor")
        
        # All results should be from the specified project
        assert all(r.project_id == "rc-quadcopter" for r in results)
    
    def test_search_all_projects(self, populated_searcher):
        """Test cross-project search."""
        results = populated_searcher.search_all_projects("motor")
        
        # Should have results from multiple projects
        projects = set(r.project_id for r in results)
        assert len(projects) > 1
    
    def test_project_filter_performance(self, populated_searcher):
        """Test that project filtering is fast (uses index)."""
        start = time.time()
        
        for _ in range(10):
            populated_searcher.search_project("rc-quadcopter", "servo")
        
        elapsed = (time.time() - start) * 1000 / 10  # Average ms per query
        
        # Should be fast due to index
        assert elapsed < 50  # 50ms is generous; should be <10ms
    
    def test_get_project_ids(self, populated_searcher):
        """Test listing all project IDs."""
        projects = populated_searcher.get_project_ids()
        
        assert "rc-quadcopter" in projects
        assert "robotic-arm" in projects
        assert "arduino-projects" in projects
    
    def test_get_project_stats(self, populated_searcher):
        """Test project statistics."""
        stats = populated_searcher.get_project_stats("rc-quadcopter")
        
        assert stats["project_id"] == "rc-quadcopter"
        assert stats["total_entries"] > 0
        assert stats["reference_count"] >= 0
        assert stats["research_count"] >= 0
        assert 0 <= stats["avg_confidence"] <= 1


class TestLinkedResearch:
    """Tests for linked research traversal."""
    
    def test_get_linked_research(self, populated_searcher):
        """Test getting linked research entries."""
        # Entry 4 has links
        links = populated_searcher.get_linked_research(4)
        
        assert len(links) > 0
        assert all(isinstance(l, LinkedResult) for l in links)
    
    def test_link_types_filter(self, populated_searcher):
        """Test filtering by link type."""
        # Get only 'supersedes' links
        links = populated_searcher.get_linked_research(4, link_types=["supersedes"])
        
        assert all(l.link_type == "supersedes" for l in links)
    
    def test_link_relevance_filter(self, populated_searcher):
        """Test filtering by relevance score."""
        all_links = populated_searcher.get_linked_research(4, relevance_min=0.0)
        high_rel_links = populated_searcher.get_linked_research(4, relevance_min=0.8)
        
        assert len(high_rel_links) <= len(all_links)
        assert all(l.relevance_score >= 0.8 for l in high_rel_links)
    
    def test_bidirectional_links(self, populated_searcher):
        """Test finding links in both directions."""
        # Entry 1 is linked FROM entry 4
        links = populated_searcher.get_linked_research(1, include_both_directions=True)
        
        # Should find the incoming link
        assert any(l.source_research_id == 4 for l in links)
    
    def test_get_superseding_research(self, populated_searcher):
        """Test finding superseding research."""
        # Entry 6 is superseded by entry 4
        newer = populated_searcher.get_superseding_research(6)
        
        # Note: This returns the link where source supersedes target
        # So searching from 6 won't find anything - we need to search from 4
        links = populated_searcher.get_linked_research(4, link_types=["supersedes"])
        assert any(l.target_research_id == 6 for l in links)


class TestConfidenceFiltering:
    """Tests for confidence-based filtering."""
    
    def test_confidence_min_filter(self, populated_searcher):
        """Test minimum confidence filtering."""
        # High confidence only
        results = populated_searcher.search("motor", confidence_min=0.85)
        
        assert all(r.confidence >= 0.85 for r in results)
    
    def test_confidence_zero_returns_all(self, populated_searcher):
        """Test that confidence_min=0 returns all results."""
        all_results = populated_searcher.search("motor", confidence_min=0.0)
        high_results = populated_searcher.search("motor", confidence_min=0.9)
        
        assert len(all_results) >= len(high_results)
    
    def test_confidence_affects_ranking(self, populated_searcher):
        """Test that higher confidence improves ranking."""
        results = populated_searcher.search("PID", debug=True)
        
        # Results with higher confidence should generally rank higher
        # (though other factors also contribute)
        if len(results) >= 2:
            # Just verify confidence component is in the ranking
            for r in results:
                if r.rank_components:
                    assert r.rank_components["confidence_component"] > 0


class TestAttachmentSearch:
    """Tests for attachment content search."""
    
    def test_search_includes_attachments(self, populated_searcher):
        """Test that search includes attachment content."""
        # Search for term that appears in attachment extracted text
        results = populated_searcher.search("sensor fusion")
        
        # Should find the MPU6050 attachment
        attachment_results = [r for r in results if r.source_type == "attachment"]
        assert len(attachment_results) > 0
    
    def test_attachment_result_attributes(self, populated_searcher):
        """Test attachment result has required attributes."""
        results = populated_searcher.search("accelerometer")
        
        attachment_results = [r for r in results if r.source_type == "attachment"]
        
        if attachment_results:
            r = attachment_results[0]
            assert r.attachment_id is not None
            assert r.attachment_filename is not None
    
    def test_exclude_attachments(self, populated_searcher):
        """Test excluding attachments from search."""
        all_results = populated_searcher.search("servo", include_attachments=True)
        no_att_results = populated_searcher.search("servo", include_attachments=False)
        
        # Without attachments should have fewer or equal results
        assert len(no_att_results) <= len(all_results)
        
        # All results should be research type
        assert all(r.source_type == "research" for r in no_att_results)


class TestFileUsage:
    """Tests for file usage tracking."""
    
    def test_get_file_usage(self, populated_searcher):
        """Test getting file usage information."""
        usage = populated_searcher.get_file_usage(1)
        
        assert isinstance(usage, FileUsage)
        assert usage.attachment_id == 1
        assert usage.filename is not None
        assert usage.total_references >= 1
    
    def test_file_usage_nonexistent(self, populated_searcher):
        """Test file usage for non-existent attachment."""
        usage = populated_searcher.get_file_usage(99999)
        
        assert usage.total_references == 0


# ==============================================================================
# PERFORMANCE TESTS
# ==============================================================================

class TestPerformance:
    """Performance tests for search operations."""
    
    def test_basic_search_latency(self, populated_searcher):
        """Test basic search is under 100ms target."""
        # Warm up
        populated_searcher.search("servo")
        
        # Measure
        latencies = []
        for _ in range(10):
            start = time.time()
            populated_searcher.search("servo")
            latencies.append((time.time() - start) * 1000)
        
        avg_latency = sum(latencies) / len(latencies)
        max_latency = max(latencies)
        
        print(f"\nBasic search latency: avg={avg_latency:.2f}ms, max={max_latency:.2f}ms")
        
        # Target: <100ms
        assert avg_latency < 100
        assert max_latency < 200  # Allow some variance
    
    def test_project_scoped_search_latency(self, populated_searcher):
        """Test project-scoped search is fast."""
        # Warm up
        populated_searcher.search_project("rc-quadcopter", "servo")
        
        # Measure
        latencies = []
        for _ in range(10):
            start = time.time()
            populated_searcher.search_project("rc-quadcopter", "servo")
            latencies.append((time.time() - start) * 1000)
        
        avg_latency = sum(latencies) / len(latencies)
        
        print(f"\nProject-scoped search latency: avg={avg_latency:.2f}ms")
        
        # Should be fast due to index
        assert avg_latency < 100
    
    def test_link_traversal_latency(self, populated_searcher):
        """Test link traversal is fast."""
        # Warm up
        populated_searcher.get_linked_research(4)
        
        # Measure
        latencies = []
        for _ in range(10):
            start = time.time()
            populated_searcher.get_linked_research(4)
            latencies.append((time.time() - start) * 1000)
        
        avg_latency = sum(latencies) / len(latencies)
        
        print(f"\nLink traversal latency: avg={avg_latency:.2f}ms")
        
        # Should be very fast
        assert avg_latency < 50


# ==============================================================================
# EDGE CASE TESTS
# ==============================================================================

class TestEdgeCases:
    """Tests for edge cases and error handling."""
    
    def test_special_characters_in_query(self, populated_searcher):
        """Test queries with special characters."""
        # Should not crash
        results = populated_searcher.search("servo-motor")
        results = populated_searcher.search("motor's")
        results = populated_searcher.search("(motor)")
        results = populated_searcher.search("motor & servo")
    
    def test_very_long_query(self, populated_searcher):
        """Test handling of very long queries."""
        long_query = " ".join(["servo"] * 100)
        results = populated_searcher.search(long_query)
        # Should not crash, may return results
    
    def test_unicode_query(self, populated_searcher):
        """Test Unicode characters in query."""
        results = populated_searcher.search("motor 电机")
        # Should not crash
    
    def test_sql_injection_prevention(self, populated_searcher):
        """Test that SQL injection is prevented."""
        # These should not cause SQL errors
        results = populated_searcher.search("'; DROP TABLE research; --")
        results = populated_searcher.search('servo" OR "1"="1')
        
        # Verify table still exists
        projects = populated_searcher.get_project_ids()
        assert len(projects) > 0
    
    def test_pagination(self, populated_searcher):
        """Test search pagination."""
        all_results = populated_searcher.search("motor", limit=100)
        page1 = populated_searcher.search("motor", limit=2, offset=0)
        page2 = populated_searcher.search("motor", limit=2, offset=2)
        
        assert len(page1) <= 2
        assert len(page2) <= 2
        
        # Pages should be different
        if len(page1) == 2 and len(page2) == 2:
            assert page1[0].research_id != page2[0].research_id


# ==============================================================================
# MAIN
# ==============================================================================

if __name__ == "__main__":
    pytest.main([__file__, "-v", "--tb=short"])
