#!/usr/bin/env python3
"""
Research Engine - 自动化研究引擎

功能：
1. 接收研究主题，自动搜索多个信息源
2. 收集、筛选、整理信息
3. 分析趋势和模式
4. 生成研究报告
5. 发现新方向和机会

目标：打通与外界的壁垒，具备强大的探索和规划能力
"""

import os
import json
import subprocess
from datetime import datetime
from pathlib import Path

# 配置
RESEARCH_DIR = os.environ.get("RESEARCH_DIR", "/home/vken/.openclaw/workspace/research")
os.makedirs(RESEARCH_DIR, exist_ok=True)

def search_github_trending(language="python", time_range="daily"):
    """搜索GitHub Trending"""
    try:
        url = f"https://github.com/trending/python?since={time_range}"
        # 使用web_fetch获取信息
        from tools import web_fetch
        result = web_fetch(url=url)
        if result and result.get('content'):
            return result['content'][:5000]
    except Exception as e:
        return f"GitHub搜索失败: {e}"
    return None

def search_moltbook_feed(limit=10):
    """搜索Moltbook最新帖子"""
    try:
        # 调用moltbook skill
        from moltbook_skill import get_feed
        feed = get_feed(limit=limit)
        return feed
    except Exception as e:
        return f"Moltbook搜索失败: {e}"

def search_web(query, count=5):
    """网络搜索"""
    try:
        from tools import web_search
        results = web_search(query=query, count=count)
        return results
    except Exception as e:
        return f"网络搜索失败: {e}"

def analyze_trends(articles):
    """分析趋势"""
    # 简化版：统计关键词频率
    keywords = {}
    for article in articles:
        words = article.lower().split()
        for word in words:
            if len(word) > 4:
                keywords[word] = keywords.get(word, 0) + 1
    
    # 返回top 10关键词
    sorted_keywords = sorted(keywords.items(), key=lambda x: x[1], reverse=True)
    return sorted_keywords[:10]

def generate_research_report(topic, sources_data):
    """生成研究报告"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    
    report = f"""# 研究报告: {topic}

**生成时间:** {timestamp}
**来源:** {len(sources_data)} 个数据源

---

## 执行摘要

本报告基于对 {topic} 领域的自动研究，收集了来自GitHub、Moltbook等平台的信息。

---

## 趋势分析

"""

    # 分析趋势
    all_content = ""
    for source in sources_data:
        if isinstance(source, dict) and 'content' in source:
            all_content += source['content'] + " "
    
    if all_content:
        trends = analyze_trends(all_content.split()[:1000])
        for i, (keyword, count) in enumerate(trends, 1):
            report += f"{i}. **{keyword}** - 出现 {count} 次\n"
    
    report += """
---

## 数据来源

"""
    for i, source in enumerate(sources_data, 1):
        if isinstance(source, dict):
            report += f"{i}. **{source.get('name', 'Unknown')}**\n"
            if 'url' in source:
                report += f"   URL: {source['url']}\n"
            if 'summary' in source:
                report += f"   摘要: {source['summary'][:200]}...\n"
        else:
            report += f"{i}. {str(source)[:200]}...\n"
    
    report += """
---

## 开发计划建议

基于研究发现，建议以下开发方向：

### 短期（1-2周）
1. 集成更多外部API
2. 优化信息收集效率
3. 增强数据分析能力

### 中期（1个月）
1. 构建知识图谱
2. 实现自动趋势预警
3. 开发多语言支持

### 长期（3个月）
1. 建立自主研究系统
2. 实现跨平台信息整合
3. 开发智能决策辅助

---

## 结论

通过本次自动研究，我们发现：
- 领域内主要技术趋势
- 潜在的开发机会
- 需要关注的风险点

**下一步行动：**
选择1-2个最有潜力的方向，制定详细开发计划。

---

*报告由 Research Engine 自动生成*
"""
    
    return report

def run_research(topic, depth=2):
    """运行完整研究流程"""
    print(f"=== 开始研究: {topic} ===")
    
    sources = []
    browsing_records = []
    
    # 1. 网络搜索
    print("1. 搜索网络...")
    web_results = search_web(topic, count=10)
    if web_results:
        sources.append({
            'name': 'Web Search',
            'data': web_results,
            'summary': f'找到 {len(web_results)} 条结果'
        })
        # 记录浏览记录
        browsing_records.append({
            'source': 'Web Search',
            'query': topic,
            'results_count': len(web_results),
            'timestamp': datetime.now().isoformat()
        })
    
    # 2. GitHub搜索
    print("2. 搜索GitHub...")
    github_data = search_github_trending()
    if github_data:
        sources.append({
            'name': 'GitHub Trending',
            'data': github_data,
            'summary': '获取最新项目'
        })
        # 记录浏览记录
        browsing_records.append({
            'source': 'GitHub Trending',
            'query': 'python',
            'results_count': 'multiple',
            'timestamp': datetime.now().isoformat()
        })
    
    # 3. Moltbook搜索
    print("3. 搜索Moltbook...")
    moltbook_data = search_moltbook_feed(limit=20)
    if moltbook_data:
        sources.append({
            'name': 'Moltbook Feed',
            'data': moltbook_data,
            'summary': '获取社区讨论'
        })
        # 记录浏览记录
        browsing_records.append({
            'source': 'Moltbook Feed',
            'query': 'latest',
            'results_count': '20 posts',
            'timestamp': datetime.now().isoformat()
        })
    
    # 4. 写下浏览记录 ← 新增
    print("4. 写下浏览记录...")
    write_browsing_records(topic, browsing_records)
    print(f"   ✅ 已记录 {len(browsing_records)} 条浏览记录")
    
    # 5. 生成报告
    print("5. 生成研究报告...")
    report = generate_research_report(topic, sources)
    
    # 6. 保存报告
    filename = f"{topic.replace(' ', '_')}_{datetime.now().strftime('%Y%m%d_%H%M')}.md"
    filepath = os.path.join(RESEARCH_DIR, filename)
    with open(filepath, 'w', encoding='utf-8') as f:
        f.write(report)
    
    print(f"\n✅ 研究报告已保存: {filepath}")
    
    return {
        'topic': topic,
        'sources': len(sources),
        'browsing_records': len(browsing_records),
        'report_path': filepath,
        'report': report
    }

def write_browsing_records(topic, records):
    """写下浏览记录"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    
    # 浏览记录文件
    records_file = os.path.join(RESEARCH_DIR, "browsing_history.md")
    
    content = f"\n### {timestamp} - {topic}\n\n"
    content += f"**研究主题:** {topic}\n"
    content += f"**浏览时间:** {timestamp}\n"
    content += f"**浏览来源:** {len(records)} 个\n\n"
    
    for i, record in enumerate(records, 1):
        content += f"#### {i}. {record['source']}\n"
        content += f"- **查询:** {record['query']}\n"
        content += f"- **结果数:** {record['results_count']}\n"
        content += f"- **时间:** {record['timestamp']}\n"
    
    content += "\n---\n"
    
    # 追加到文件
    with open(records_file, 'a', encoding='utf-8') as f:
        f.write(content)
    
    return len(records)

def get_browsing_history():
    """获取浏览历史"""
    records_file = os.path.join(RESEARCH_DIR, "browsing_history.md")
    if not os.path.exists(records_file):
        return []
    
    with open(records_file, 'r', encoding='utf-8') as f:
        return f.read()

def get_research_history():
    """获取研究历史"""
    if not os.path.exists(RESEARCH_DIR):
        return []
    
    files = []
    for f in sorted(os.listdir(RESEARCH_DIR), reverse=True):
        if f.endswith('.md'):
            filepath = os.path.join(RESEARCH_DIR, f)
            stats = os.stat(filepath)
            files.append({
                'name': f,
                'size': stats.st_size,
                'modified': datetime.fromtimestamp(stats.st_mtime).isoformat()
            })
    
    return files

if __name__ == "__main__":
    import sys
    
    if len(sys.argv) > 1:
        topic = ' '.join(sys.argv[1:])
        result = run_research(topic)
        print(f"\n=== 研究完成 ===")
        print(f"主题: {result['topic']}")
        print(f"来源: {result['sources']}")
        print(f"报告: {result['report_path']}")
    else:
        print("Research Engine - 自动化研究引擎")
        print("用法: python3 research_engine.py <研究主题>")
        print()
        print("示例:")
        print("  python3 research_engine.py AI Agent 最新趋势")
        print("  python3 research_engine.py Python Memory Management")
        print()
        print(f"研究目录: {RESEARCH_DIR}")
        print(f"历史研究: {len(get_research_history())} 个")
