#!/usr/bin/env python3
"""
report.py - Generate markdown triage reports from scan.py JSON output.

Usage:
    python3 report.py <scores.json> [--output-dir <dir>]

Generates:
    - prioritize.md  (score 80+)
    - review.md      (score 50-79)
    - close.md       (score <50)
    - summary.md     (overview with distribution, top 3, patterns, duplicates)
"""

import argparse
import json
import sys
from collections import Counter
from pathlib import Path


def load_scores(path: str) -> dict:
    """Load the JSON output from scan.py."""
    try:
        return json.loads(Path(path).read_text())
    except (json.JSONDecodeError, FileNotFoundError) as e:
        print(f"Error loading scores: {e}", file=sys.stderr)
        sys.exit(1)


def pr_to_markdown(pr: dict, repo: str) -> str:
    """Format a single PR as a markdown section."""
    url = f"https://github.com/{repo}/pull/{pr['number']}"
    mods = ", ".join(f"{m['modifier']} ({m['points']:+d})" for m in pr.get("modifiers", []))
    labels = ", ".join(pr.get("labels", [])) or "none"
    stats = pr.get("stats", {})

    return f"""### [{pr['title']}]({url}) (#{pr['number']})
- **Score:** {pr['score']} | **Action:** {pr['action']}
- **Author:** {pr['author']} | **Created:** {pr.get('created_at', 'unknown')[:10]}
- **Stats:** +{stats.get('additions', 0)} / -{stats.get('deletions', 0)} across {stats.get('changed_files', 0)} files
- **Labels:** {labels}
- **Modifiers:** {mods or 'none'}
"""


def generate_report(prs: list[dict], repo: str, title: str, description: str) -> str:
    """Generate a markdown report for a category of PRs."""
    if not prs:
        return f"# {title}\n\n{description}\n\n*No PRs in this category.*\n"

    lines = [f"# {title}\n", f"{description}\n", f"**Count:** {len(prs)}\n"]
    for pr in prs:
        lines.append(pr_to_markdown(pr, repo))
    return "\n".join(lines)


def generate_summary(data: dict) -> str:
    """Generate the summary report."""
    repo = data["repo"]
    scored = data["scored_prs"]
    dist = data["distribution"]
    dupes = data.get("potential_duplicates", [])

    # Top 3
    top3 = scored[:3]
    top3_lines = "\n".join(
        f"{i+1}. **#{pr['number']}** - {pr['title']} (score: {pr['score']})"
        for i, pr in enumerate(top3)
    )

    # Common modifiers (patterns)
    all_mods = []
    for pr in scored:
        for m in pr.get("modifiers", []):
            all_mods.append(m["modifier"])
    common = Counter(all_mods).most_common(10)
    pattern_lines = "\n".join(f"- {mod}: {count} PRs" for mod, count in common)

    # Score distribution
    brackets = {
        "90-100": len([p for p in scored if p["score"] >= 90]),
        "80-89": len([p for p in scored if 80 <= p["score"] < 90]),
        "70-79": len([p for p in scored if 70 <= p["score"] < 80]),
        "60-69": len([p for p in scored if 60 <= p["score"] < 70]),
        "50-59": len([p for p in scored if 50 <= p["score"] < 60]),
        "40-49": len([p for p in scored if 40 <= p["score"] < 50]),
        "30-39": len([p for p in scored if 30 <= p["score"] < 40]),
        "0-29": len([p for p in scored if p["score"] < 30]),
    }
    dist_lines = "\n".join(f"| {bracket} | {count} |" for bracket, count in brackets.items())

    # Duplicates
    if dupes:
        dupe_lines = "\n".join(
            f"- #{d['pr_a']} and #{d['pr_b']} (similarity: {d['similarity']:.0%})"
            for d in dupes
        )
    else:
        dupe_lines = "No potential duplicates found."

    # Authors
    authors = Counter(pr["author"] for pr in scored)
    prolific = authors.most_common(5)
    author_lines = "\n".join(f"- {a}: {c} PRs" for a, c in prolific)

    return f"""# PR Triage Summary: {repo}

## Overview
- **Total PRs scored:** {data['total_prs']}
- **Prioritize (80+):** {dist['prioritize']}
- **Review (50-79):** {dist['review']}
- **Close (<50):** {dist['close']}

## Top 3 PRs
{top3_lines}

## Score Distribution

| Range | Count |
|-------|-------|
{dist_lines}

## Common Patterns
{pattern_lines}

## Potential Duplicates
{dupe_lines}

## Most Active Authors
{author_lines}

---
*Generated by pr-triage scan. Review the individual reports for details.*
"""


def main():
    parser = argparse.ArgumentParser(description="Generate triage reports from scan results")
    parser.add_argument("scores_file", help="Path to JSON output from scan.py")
    parser.add_argument("--output-dir", default="./triage-reports",
                        help="Directory for output markdown files (default: ./triage-reports)")
    args = parser.parse_args()

    data = load_scores(args.scores_file)
    repo = data["repo"]
    scored = data["scored_prs"]
    output_dir = Path(args.output_dir)
    output_dir.mkdir(parents=True, exist_ok=True)

    # Split into categories
    prioritize = [p for p in scored if p["score"] >= 80]
    review = [p for p in scored if 50 <= p["score"] < 80]
    close = [p for p in scored if p["score"] < 50]

    # Generate reports
    reports = {
        "prioritize.md": generate_report(
            prioritize, repo,
            "Prioritize: High-Alignment PRs (80+)",
            "These PRs strongly align with the project vision. Fast-track for review."
        ),
        "review.md": generate_report(
            review, repo,
            "Review: Standard Queue PRs (50-79)",
            "These PRs are reasonable contributions. Review when bandwidth allows."
        ),
        "close.md": generate_report(
            close, repo,
            "Close: Misaligned PRs (<50)",
            "These PRs do not align well with the project vision. Consider closing with explanation."
        ),
        "summary.md": generate_summary(data),
    }

    for filename, content in reports.items():
        path = output_dir / filename
        path.write_text(content)
        print(f"Written: {path}", file=sys.stderr)

    print(f"\nReports generated in {output_dir}/", file=sys.stderr)


if __name__ == "__main__":
    main()
