---
name: remarkable
description: Bidirectional sync with reMarkable tablet via Cloud API (rmapi). Fetch handwritten notes/sketches, process with AI, and push content back. Use for sketch enhancement, journal extraction, or sending documents/images to the tablet.
---

# reMarkable Tablet Integration (rmapi)

Bidirectional sync with reMarkable tablet via Cloud API. Fetch sketches and notes, process them (AI enhancement, text extraction), and push content back to the tablet.

## Typical Use Cases

1. **Sketch → AI → Tablet loop** — Fetch rough sketch → enhance with AI → push polished version back
2. **Journal entries** — Fetch handwritten thoughts → interpret → append to memory/journal
3. **Brainstorming** — Fetch diagrams/lists → extract structure → add to project docs
4. **Send reading material** — Push PDFs/documents to tablet for offline reading
5. **AI art to tablet** — Generate images → convert to PDF → push for viewing on e-ink

## Bidirectional Pipeline

```
┌─────────────────────────────────────────────────────────────────┐
│                        FETCH (tablet → agent)                    │
├─────────────────────────────────────────────────────────────────┤
│  reMarkable → Cloud sync → rmapi get → .rmdoc                   │
│                                           ↓                      │
│                              unzip → .rm file → rmc → SVG       │
│                                                      ↓           │
│                                          cairosvg → PNG          │
│                                                      ↓           │
│                               ┌──────────┴──────────┐            │
│                          Text content?         Visual/sketch?    │
│                               ↓                      ↓           │
│                        OCR/interpret          AI image editing   │
│                               ↓                      ↓           │
│                        Add to memory        Enhanced image       │
└─────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────┐
│                        PUSH (agent → tablet)                     │
├─────────────────────────────────────────────────────────────────┤
│  Image/document → Convert to PDF (if needed) → rmapi put        │
│                                                      ↓           │
│                                           Cloud sync → tablet    │
└─────────────────────────────────────────────────────────────────┘
```

## Prerequisites

### 1. Install rmapi

Download the latest release from [juruen/rmapi](https://github.com/ddvk/rmapi/releases) (ddvk fork recommended):

```bash
# Example for Linux amd64
curl -L https://github.com/ddvk/rmapi/releases/latest/download/rmapi-linux-amd64 -o ~/bin/rmapi
chmod +x ~/bin/rmapi
```

### 2. Install conversion tools

```bash
pip install --user rmc cairosvg pillow
```

- **rmc** — Converts `.rm` stroke files to SVG
- **cairosvg** — Converts SVG to PNG
- **pillow** — Converts PNG to PDF for pushing back

### 3. Authenticate (ONE-TIME)

1. Go to https://my.remarkable.com/connect/desktop
2. Log in and copy the 8-character code
3. Run `rmapi` and paste the code when prompted
4. Token saved to `~/.rmapi` — future runs are automatic

## Commands

### Fetch (Download from tablet)

```bash
# List files/folders
rmapi ls
rmapi ls -l -t              # Long format, sorted by time

# Refresh from cloud
rmapi refresh

# Navigate
rmapi cd "folder name"

# Find files
rmapi find --starred /              # Starred items
rmapi find --tag="my-tag" /         # By tag
rmapi find / ".*sketch.*"           # By regex

# Download single file
rmapi get "filename"
rmapi get "Folder/Notebook"

# Download with annotations (best for handwritten content)
rmapi geta "filename"

# Bulk download folder
rmapi mget -o ./sync-folder/ "/My Folder"
```

### Push (Upload to tablet)

```bash
# Upload single file (PDF or EPUB only)
rmapi put document.pdf
rmapi put document.pdf "Target Folder/"

# Bulk upload
rmapi mput ./local-folder/ "Remote Folder/"

# Create folder on tablet
rmapi mkdir "New Folder"
```

**Supported formats:** PDF, EPUB

## Conversion Workflows

### Fetch: .rmdoc → PNG

reMarkable notebooks download as `.rmdoc` (a ZIP archive containing stroke data):

```bash
# 1. Download the notebook
rmapi get "Folder/MyNotebook"

# 2. Extract (it's a zip)
unzip "MyNotebook.rmdoc" -d extracted/

# 3. Find the .rm stroke file(s)
# Structure: extracted/<doc-uuid>/<page-uuid>.rm
find extracted -name "*.rm"

# 4. Convert .rm → SVG
rmc -t svg -o page.svg "extracted/<doc-uuid>/<page-uuid>.rm"

# 5. Convert SVG → PNG (reMarkable dimensions: 1404×1872)
python3 -c "
import cairosvg
cairosvg.svg2png(url='page.svg', write_to='page.png', output_width=1404, output_height=1872)
"
```

### Push: Image → PDF

reMarkable only accepts PDF/EPUB, so convert images first:

```python
from PIL import Image

img = Image.open('image.png')
rgb = img.convert('RGB')
rgb.save('image.pdf', 'PDF', resolution=150)
```

Then push:
```bash
rmapi put image.pdf "My Folder/"
```

## Full Workflow Example

**Sketch enhancement loop:**

```bash
# 1. Fetch sketch from tablet
rmapi get "Sketches/MyDrawing"

# 2. Extract and convert to PNG
unzip MyDrawing.rmdoc -d MyDrawing_extracted/
RM_FILE=$(find MyDrawing_extracted -name "*.rm" | head -1)
rmc -t svg -o sketch.svg "$RM_FILE"
python3 -c "import cairosvg; cairosvg.svg2png(url='sketch.svg', write_to='sketch.png', output_width=1404, output_height=1872)"

# 3. [Your AI enhancement step here]
# Example: use any image-to-image AI tool to enhance sketch.png → enhanced.png

# 4. Convert to PDF and push back
python3 -c "from PIL import Image; Image.open('enhanced.png').convert('RGB').save('enhanced.pdf', 'PDF', resolution=150)"
rmapi put enhanced.pdf "Sketches/"
```

## Sharing Strategies

Create a dedicated sync folder on your tablet, or use:
- **Tags** — Tag items for discovery with `rmapi find --tag`
- **Stars** — Star items for quick access with `rmapi find --starred`

## Notes

- **Cloud sync required** — Tablet must sync to cloud before files are available (pull down to refresh on tablet)
- **Format** — `.rmdoc` is a ZIP containing JSON metadata + `.rm` binary stroke files
- **Warnings** — `rmc` may show warnings about newer format versions — usually still works
- **Dimensions** — reMarkable display is 1404×1872 pixels (portrait)
- **Text extraction** — For handwritten text, use vision models to interpret visually rather than traditional OCR

## Troubleshooting

| Issue | Solution |
|-------|----------|
| `rmapi` not connecting | Re-authenticate: delete `~/.rmapi` and run `rmapi` again |
| File not found after upload | Wait for cloud sync, or refresh tablet manually |
| `rmc` format warnings | Usually safe to ignore; output still generated |
| SVG looks empty | Check if the correct `.rm` file was used (multi-page notebooks have multiple) |
