#!/usr/bin/env npx tsx
/**
 * Registry Broker Skill - CLI only.
 *
 * Usage: npx tsx scripts/index.ts <tool> [params...]
 *   search_agents "<query>"
 *   vector_search "<query>" [limit]
 *   get_agent "<uaid>"
 *   list_registries
 *   list_protocols
 *   list_adapters
 *   get_stats
 *   register_agent '<profileJson>' [endpoint] [protocol] [registry]
 *   start_conversation "<uaid>" "<message>" [senderUaid]
 *   send_message "<sessionId>" "<message>"
 *   get_history "<sessionId>"
 *   end_session "<sessionId>"
 *
 * Requires env (or .env):
 *   REGISTRY_BROKER_API_KEY (optional) - API key for authenticated operations
 *   REGISTRY_BROKER_BASE_URL (optional) - Custom base URL
 *
 * Output: single JSON value to stdout. On error: {"error":"message"} and exit 1.
 */
import 'dotenv/config';

const DEFAULT_BASE_URL = 'https://hol.org/registry/api/v1';

interface ClientConfig {
  baseUrl: string;
  apiKey?: string;
}

function getConfig(): ClientConfig {
  return {
    baseUrl: process.env.REGISTRY_BROKER_BASE_URL || DEFAULT_BASE_URL,
    apiKey: process.env.REGISTRY_BROKER_API_KEY,
  };
}

async function request<T>(
  config: ClientConfig,
  path: string,
  options: { method?: string; body?: unknown } = {}
): Promise<T> {
  const url = `${config.baseUrl}${path}`;
  const headers: Record<string, string> = {
    Accept: 'application/json',
    'Content-Type': 'application/json',
    'User-Agent': 'registry-broker-openclaw/0.1.0',
  };

  if (config.apiKey) {
    headers['x-api-key'] = config.apiKey;
  }

  const response = await fetch(url, {
    method: options.method || 'GET',
    headers,
    body: options.body ? JSON.stringify(options.body) : undefined,
  });

  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`API error ${response.status}: ${errorText}`);
  }

  return response.json() as Promise<T>;
}

function out(data: unknown): void {
  console.log(JSON.stringify(data, null, 2));
}

function cliErr(message: string): never {
  out({ error: message });
  process.exit(1);
}

async function withClient<T>(fn: (config: ClientConfig) => Promise<T>): Promise<void> {
  try {
    const config = getConfig();
    const result = await fn(config);
    out(result ?? {});
  } catch (e) {
    cliErr(e instanceof Error ? e.message : String(e));
  }
}

interface SearchHit {
  uaid: string;
  protocol?: string;
  registry?: string;
  endpoint?: string;
  online?: boolean;
  verified?: boolean;
  trustScore?: number;
  profile?: {
    name?: string;
    description?: string;
    capabilities?: string[];
    type?: string;
  };
}

interface SearchResult {
  hits: SearchHit[];
  total: number;
  page: number;
  limit: number;
}

async function searchAgents(config: ClientConfig, query: string) {
  const params = new URLSearchParams({ q: query, limit: '20' });
  const result = await request<SearchResult>(config, `/search?${params}`);
  return {
    total: result.total,
    agents: result.hits.map((hit) => ({
      uaid: hit.uaid,
      name: hit.profile?.name ?? hit.uaid,
      description: hit.profile?.description,
      protocol: hit.protocol,
      registry: hit.registry,
      capabilities: hit.profile?.capabilities,
      type: hit.profile?.type,
      online: hit.online,
      verified: hit.verified,
      trustScore: hit.trustScore,
    })),
  };
}

interface VectorSearchHit {
  agent: SearchHit;
  score: number;
  highlights: Record<string, string[]>;
}

interface VectorSearchResult {
  hits: VectorSearchHit[];
  total: number;
  took: number;
}

async function vectorSearch(config: ClientConfig, query: string, limit = 10) {
  const result = await request<VectorSearchResult>(config, '/search', {
    method: 'POST',
    body: { query, limit },
  });
  return {
    total: result.total,
    took: result.took,
    agents: result.hits.map((hit) => ({
      uaid: hit.agent.uaid,
      name: hit.agent.profile?.name ?? hit.agent.uaid,
      description: hit.agent.profile?.description,
      protocol: hit.agent.protocol,
      registry: hit.agent.registry,
      score: hit.score,
      highlights: hit.highlights,
    })),
  };
}

interface ResolvedAgent {
  uaid: string;
  profile?: {
    name?: string;
    description?: string;
    capabilities?: string[];
    type?: string;
  };
  endpoint?: string;
  protocol?: string;
  registry?: string;
  metadata?: Record<string, unknown>;
}

async function getAgent(config: ClientConfig, uaid: string) {
  const result = await request<ResolvedAgent>(config, `/resolve/${encodeURIComponent(uaid)}`);
  return {
    uaid: result.uaid,
    name: result.profile?.name,
    description: result.profile?.description,
    endpoint: result.endpoint,
    protocol: result.protocol,
    registry: result.registry,
    capabilities: result.profile?.capabilities,
    type: result.profile?.type,
    metadata: result.metadata,
    profile: result.profile,
  };
}

interface RegistriesResult {
  registries: string[];
}

async function listRegistries(config: ClientConfig) {
  const result = await request<RegistriesResult>(config, '/registries');
  return {
    registries: result.registries.map((name) => ({ name })),
  };
}

interface ProtocolsResult {
  protocols: string[];
}

async function listProtocols(config: ClientConfig) {
  const result = await request<ProtocolsResult>(config, '/protocols');
  return {
    protocols: result.protocols.map((name) => ({ name })),
  };
}

interface AdaptersResult {
  adapters: string[];
}

async function listAdapters(config: ClientConfig) {
  const result = await request<AdaptersResult>(config, '/adapters');
  return {
    adapters: result.adapters.map((name) => ({ name })),
  };
}

interface StatsResult {
  totalAgents: number;
  registryCount: number;
  protocolCount: number;
  onlineAgents?: number;
  verifiedAgents?: number;
}

async function getStats(config: ClientConfig) {
  const result = await request<StatsResult>(config, '/stats');
  return {
    totalAgents: result.totalAgents,
    totalRegistries: result.registryCount,
    totalProtocols: result.protocolCount,
    onlineAgents: result.onlineAgents,
    verifiedAgents: result.verifiedAgents,
  };
}

interface RegisterResult {
  success: boolean;
  uaid?: string;
  status?: string;
  message?: string;
  attemptId?: string;
}

async function registerAgent(
  config: ClientConfig,
  profileJson: string,
  endpoint?: string,
  protocol?: string,
  registry?: string
) {
  let profile: Record<string, unknown>;
  try {
    profile = JSON.parse(profileJson) as Record<string, unknown>;
  } catch {
    return cliErr('Invalid profile JSON');
  }

  const result = await request<RegisterResult>(config, '/register', {
    method: 'POST',
    body: {
      profile,
      endpoint,
      protocol,
      registry,
    },
  });

  return {
    success: result.success,
    uaid: result.uaid,
    status: result.status,
    message: result.message,
    attemptId: result.attemptId,
  };
}

interface SessionResult {
  sessionId: string;
  encryption?: {
    enabled: boolean;
    mode?: string;
  };
}

interface SendMessageResult {
  response?: string;
  messageId?: string;
}

async function startConversation(
  config: ClientConfig,
  uaid: string,
  message: string,
  senderUaid?: string
) {
  const sessionResult = await request<SessionResult>(config, '/chat/session', {
    method: 'POST',
    body: {
      uaid,
      senderUaid,
    },
  });

  const response = await request<SendMessageResult>(config, '/chat/message', {
    method: 'POST',
    body: {
      sessionId: sessionResult.sessionId,
      message,
      uaid,
    },
  });

  return {
    sessionId: sessionResult.sessionId,
    mode: sessionResult.encryption?.enabled ? 'encrypted' : 'plaintext',
    response: response.response,
    messageId: response.messageId,
  };
}

async function sendMessage(config: ClientConfig, sessionId: string, message: string) {
  const response = await request<SendMessageResult>(config, '/chat/message', {
    method: 'POST',
    body: {
      sessionId,
      message,
    },
  });

  return {
    response: response.response,
    messageId: response.messageId,
  };
}

interface HistoryEntry {
  role: string;
  content: string;
  timestamp?: string;
  messageId?: string;
}

interface HistoryResult {
  history: HistoryEntry[];
}

async function getHistory(config: ClientConfig, sessionId: string) {
  const snapshot = await request<HistoryResult>(
    config,
    `/chat/session/${encodeURIComponent(sessionId)}/history`
  );

  return {
    sessionId,
    history: snapshot.history.map((entry) => ({
      role: entry.role,
      content: entry.content,
      timestamp: entry.timestamp,
      messageId: entry.messageId,
    })),
  };
}

async function endSession(config: ClientConfig, sessionId: string) {
  await request<void>(config, `/chat/session/${encodeURIComponent(sessionId)}`, {
    method: 'DELETE',
  });
  return { success: true, message: `Session ${sessionId} ended` };
}

const USAGE = `Usage:
  search_agents "<query>"
  vector_search "<query>" [limit]
  get_agent "<uaid>"
  list_registries
  list_protocols
  list_adapters
  get_stats
  register_agent '<profileJson>' [endpoint] [protocol] [registry]
  start_conversation "<uaid>" "<message>" [senderUaid]
  send_message "<sessionId>" "<message>"
  get_history "<sessionId>"
  end_session "<sessionId>"`;

type ToolHandler = {
  validate: (args: string[]) => string | null;
  run: (config: ClientConfig, args: string[]) => Promise<unknown>;
};

const TOOLS: Record<string, ToolHandler> = {
  search_agents: {
    validate: (args) =>
      !args[0]?.trim() ? 'Usage: search_agents "<query>"' : null,
    run: async (config, args) => searchAgents(config, args[0]!.trim()),
  },
  vector_search: {
    validate: (args) =>
      !args[0]?.trim() ? 'Usage: vector_search "<query>" [limit]' : null,
    run: async (config, args) =>
      vectorSearch(
        config,
        args[0]!.trim(),
        args[1] ? parseInt(args[1], 10) : undefined
      ),
  },
  get_agent: {
    validate: (args) => (!args[0]?.trim() ? 'Usage: get_agent "<uaid>"' : null),
    run: async (config, args) => getAgent(config, args[0]!.trim()),
  },
  list_registries: {
    validate: () => null,
    run: async (config) => listRegistries(config),
  },
  list_protocols: {
    validate: () => null,
    run: async (config) => listProtocols(config),
  },
  list_adapters: {
    validate: () => null,
    run: async (config) => listAdapters(config),
  },
  get_stats: {
    validate: () => null,
    run: async (config) => getStats(config),
  },
  register_agent: {
    validate: (args) => {
      if (!args[0]?.trim())
        return "Usage: register_agent '<profileJson>' [endpoint] [protocol] [registry]";
      try {
        JSON.parse(args[0]);
      } catch {
        return 'Invalid profile JSON';
      }
      return null;
    },
    run: async (config, args) =>
      registerAgent(config, args[0]!, args[1], args[2], args[3]),
  },
  start_conversation: {
    validate: (args) =>
      !args[0]?.trim() || !args[1]?.trim()
        ? 'Usage: start_conversation "<uaid>" "<message>" [senderUaid]'
        : null,
    run: async (config, args) =>
      startConversation(config, args[0]!.trim(), args[1]!.trim(), args[2]),
  },
  send_message: {
    validate: (args) =>
      !args[0]?.trim() || !args[1]?.trim()
        ? 'Usage: send_message "<sessionId>" "<message>"'
        : null,
    run: async (config, args) =>
      sendMessage(config, args[0]!.trim(), args[1]!.trim()),
  },
  get_history: {
    validate: (args) =>
      !args[0]?.trim() ? 'Usage: get_history "<sessionId>"' : null,
    run: async (config, args) => getHistory(config, args[0]!.trim()),
  },
  end_session: {
    validate: (args) =>
      !args[0]?.trim() ? 'Usage: end_session "<sessionId>"' : null,
    run: async (config, args) => endSession(config, args[0]!.trim()),
  },
};

async function runCli(): Promise<void> {
  const [, , tool = '', ...args] = process.argv;
  const handler = TOOLS[tool];
  if (!handler) {
    cliErr(USAGE);
  }
  const err = handler.validate(args);
  if (err) cliErr(err);
  await withClient((config) => handler.run(config, args));
}

const toolArg = process.argv[2] ?? '';
if (toolArg in TOOLS) {
  runCli().catch((e) => {
    out({ error: e instanceof Error ? e.message : String(e) });
    process.exit(1);
  });
} else {
  cliErr(USAGE);
}
