# Security Audit Report: receipts-guard v0.6.0

**Audit Date:** 2026-02-09
**Auditor:** Claude Code
**Version:** 0.6.0 (Self-Sovereign Agent Identity)

---

## Executive Summary

receipts-guard v0.6.0 introduces Ed25519-based cryptographic signatures and DID (Decentralized Identifier) identity. This audit found **1 security issue (FIXED)** and verified the cryptographic implementation is sound.

### Overall Assessment: **PASS** ✅

---

## Test Results

### QA Tests (12/12 Passed)

| Test | Status |
|------|--------|
| Identity Show | ✅ Pass |
| Identity Show --full | ✅ Pass |
| Identity Init (duplicate prevention) | ✅ Pass |
| DID Verification (valid) | ✅ Pass |
| DID Verification (invalid format) | ✅ Pass |
| DID Verification (non-existent) | ✅ Pass |
| Proposal validation (empty terms) | ✅ Pass |
| Proposal validation (missing counterparty) | ✅ Pass |
| Accept (non-existent proposal) | ✅ Pass |
| Fulfill (non-existent agreement) | ✅ Pass |
| List with filters | ✅ Pass |
| Legacy backward compatibility | ✅ Pass |

### Cryptographic Tests (6/6 Passed)

| Test | Status |
|------|--------|
| Ed25519 signature creation | ✅ Pass |
| Ed25519 signature verification (valid) | ✅ Pass |
| Signature verification (wrong terms hash) | ✅ Correctly rejected |
| Signature verification (modified timestamp) | ✅ Correctly rejected |
| Signature verification (tampered signature) | ✅ Correctly rejected |
| Signature verification (truncated) | ✅ Correctly rejected |

### Security Tests (10/10 Passed)

| Test | Status |
|------|--------|
| Private key permissions (600) | ✅ Pass |
| Private directory permissions (700) | ✅ Pass (after fix) |
| Archived key permissions (600) | ✅ Pass (after fix) |
| Path traversal in terms | ✅ Contained (treated as data) |
| JSON injection in terms | ✅ Contained (escaped properly) |
| Command injection in counterparty | ✅ Contained (not executed) |
| Key rotation chain integrity | ✅ Valid |
| DID document structure | ✅ Valid |
| Key-history consistency | ✅ Valid |
| Signature replay prevention | ✅ Pass (timestamp binding) |

---

## Issues Found and Resolved

### SEC-001: Archived Key File Permissions (FIXED)

**Severity:** Medium
**Status:** Fixed

**Description:**
Archived private keys in `~/.openclaw/receipts/identity/private/key-archive/` were created with default permissions (644), making them world-readable.

**Impact:**
Other users on a shared system could potentially read old private keys.

**Fix Applied:**
```javascript
// In handleIdentityRotate() - added chmod after archive write
const archiveFilePath = path.join(KEY_ARCHIVE_DIR, `${currentKeyData.keyId.replace('#', '')}.json`);
fs.writeFileSync(archiveFilePath, JSON.stringify({...}, null, 2));
try {
  fs.chmodSync(archiveFilePath, 0o600); // Owner read/write only
} catch (e) {}
```

Also updated `ensureDir()` to support restricted mode:
```javascript
function ensureDir(dirPath, restrictedMode = false) {
  if (!fs.existsSync(dirPath)) {
    fs.mkdirSync(dirPath, { recursive: true });
  }
  if (restrictedMode) {
    try {
      fs.chmodSync(dirPath, 0o700); // Owner only: rwx------
    } catch (e) {}
  }
}
```

---

## Security Architecture Analysis

### Key Storage Model

```
~/.openclaw/receipts/identity/
├── did.json                 [644] Public DID document
├── controller.json          [644] Controller metadata
├── key-history.json         [644] Public key history
├── private/                 [700] Restricted directory
│   ├── key-current.json     [600] Active private key
│   └── key-archive/         [700] Restricted directory
│       ├── key-*.json       [600] Rotated private keys
└── recovery/                [700] Recovery tokens (future)
```

### Cryptographic Properties

| Property | Implementation | Status |
|----------|---------------|--------|
| Algorithm | Ed25519 (EdDSA) | ✅ Industry standard |
| Key size | 256-bit | ✅ Secure |
| Message binding | `hash|timestamp` | ✅ Prevents replay |
| Encoding | Base64url (signatures), Base58btc (keys) | ✅ Standard |
| Key rotation | Old key signs new key | ✅ Verifiable chain |

### Signature Format

```
ed25519:<base64url_signature>:<unix_timestamp_ms>
```

The timestamp is included in the signed message, preventing:
- Signature replay attacks
- Timestamp manipulation
- Cross-message signature reuse

### DID Document Security

- **W3C DID Core compliant** structure
- **Key history preserved** for historical signature verification
- **Human controller backstop** for key compromise recovery
- **No external dependencies** for local identity

---

## Recommendations

### Implemented in v0.6.0
1. ✅ Ed25519 signatures (replacing HMAC)
2. ✅ DID-based identity
3. ✅ Key rotation with proof chain
4. ✅ Restricted file permissions

### Recommended for Future Versions

1. **Key Encryption at Rest** (v0.7.0)
   - Encrypt private keys with passphrase
   - Use PBKDF2/Argon2 for key derivation

2. **DID Registry Integration** (v0.8.0)
   - Publish DIDs to decentralized registry
   - Enable cross-agent DID resolution

3. **Hardware Key Support** (v0.9.0)
   - Support for hardware security modules
   - FIDO2/WebAuthn integration

4. **Signature Expiration** (Consider)
   - Add optional validity period to signatures
   - Require re-signing for long-term agreements

---

## Test Commands

```bash
# Verify identity
node capture.js identity show --full

# Test signature verification
node capture.js identity verify --signature="<sig>" --termsHash="<hash>"

# Check file permissions
ls -la ~/.openclaw/receipts/identity/private/
stat -f "%Sp %OLp" ~/.openclaw/receipts/identity/private/key-current.json
```

---

## Conclusion

receipts-guard v0.6.0 implements a robust self-sovereign identity system based on W3C DID standards and Ed25519 cryptography. The single security issue found (file permissions) has been fixed. The system correctly:

- Generates secure Ed25519 keypairs
- Signs agreements with timestamp binding
- Verifies signatures against DID documents
- Maintains key rotation chain integrity
- Protects private keys with filesystem permissions

**Recommendation:** Approved for production use.

---

*This audit was performed by Claude Code as part of the receipts-guard development process.*
