#!/bin/bash

# React Next.js 项目生成器脚本

PROJECT_NAME=$1
OUTPUT_DIR=$2

if [ -z "$PROJECT_NAME" ] || [ -z "$OUTPUT_DIR" ]; then
  echo "用法: $0 <项目名称> <输出目录>"
  exit 1
fi

echo "正在创建项目: $PROJECT_NAME"
echo "输出目录: $OUTPUT_DIR"

# 创建项目目录结构
mkdir -p "$OUTPUT_DIR/$PROJECT_NAME"
cd "$OUTPUT_DIR/$PROJECT_NAME"

# 初始化Next.js项目
npx create-next-app@latest . --typescript --tailwind --eslint --app --src-dir --import-alias "@/*"

# 安装额外依赖
npm install antd @ant-design/icons zustand

# 更新package.json以包含更多必要的依赖
cat > package.json << EOF
{
  "name": "$PROJECT_NAME",
  "version": "0.1.0",
  "private": true,
  "scripts": {
    "dev": "next dev",
    "build": "next build",
    "start": "next start",
    "lint": "next lint"
  },
  "dependencies": {
    "react": "^18",
    "react-dom": "^18",
    "next": "14.0.1",
    "antd": "^5.0.0",
    "zustand": "^4.4.0",
    "@ant-design/icons": "^5.0.0"
  },
  "devDependencies": {
    "typescript": "^5",
    "@types/node": "^20",
    "@types/react": "^18",
    "@types/react-dom": "^18",
    "autoprefixer": "^10.4.16",
    "postcss": "^8",
    "tailwindcss": "^3.3.0",
    "eslint": "^8",
    "eslint-config-next": "14.0.1"
  }
}
EOF

# 创建Zustand store示例
mkdir -p src/stores
cat > src/stores/useCounterStore.ts << EOF
import { create } from 'zustand';

interface CounterState {
  count: number;
  increment: () => void;
  decrement: () => void;
  reset: () => void;
}

export const useCounterStore = create<CounterState>((set) => ({
  count: 0,
  increment: () => set((state) => ({ count: state.count + 1 })),
  decrement: () => set((state) => ({ count: state.count - 1 })),
  reset: () => set({ count: 0 }),
}));
EOF

# 创建Ant Design和Tailwind集成的布局组件
mkdir -p src/components
cat > src/components/Layout.tsx << EOF
import React, { ReactNode } from 'react';
import Head from 'next/head';
import { Layout as AntLayout, Menu, theme } from 'antd';
import { AppstoreOutlined, MailOutlined, SettingOutlined } from '@ant-design/icons';

const { Header, Content, Footer } = AntLayout;

interface LayoutProps {
  children: ReactNode;
  title?: string;
}

const MainLayout: React.FC<LayoutProps> = ({ children, title = 'My App' }) => {
  const {
    token: { colorBgContainer },
  } = theme.useToken();

  return (
    <div className="ant-layout">
      <Head>
        <title>{title}</title>
        <meta name="description" content="Generated by create next app" />
        <link rel="icon" href="/favicon.ico" />
      </Head>
      
      <Header className="header bg-blue-600 text-white">
        <div className="logo text-xl font-bold">LOGO</div>
        <Menu
          theme="dark"
          mode="horizontal"
          defaultSelectedKeys={['2']}
          items={[
            {
              key: '1',
              label: 'nav 1',
              icon: <AppstoreOutlined />,
            },
            {
              key: '2',
              label: 'nav 2',
              icon: <MailOutlined />,
            },
            {
              key: '3',
              label: 'nav 3',
              icon: <SettingOutlined />,
            },
          ]}
          className="bg-blue-600"
        />
      </Header>
      
      <Content style={{ padding: '0 50px' }}>
        <div className="bg-gray-100 rounded-lg p-6 min-h-[calc(100vh-230px)] mt-4">
          {children}
        </div>
      </Content>
      
      <Footer style={{ textAlign: 'center' }} className="mt-10">
        Ant Design ©{new Date().getFullYear()} Created by Ant UED
      </Footer>
    </div>
  );
};

export default MainLayout;
EOF

# 更新主页面以使用布局
cat > src/app/page.tsx << EOF
'use client';
import React from 'react';
import { Button, Card, Space, Typography } from 'antd';
import { PlusOutlined, MinusOutlined, ReloadOutlined } from '@ant-design/icons';
import MainLayout from '@/components/Layout';
import { useCounterStore } from '@/stores/useCounterStore';

const { Title, Text } = Typography;

export default function Home() {
  const { count, increment, decrement, reset } = useCounterStore();

  return (
    <MainLayout title="首页">
      <Card className="w-full max-w-md mx-auto shadow-lg">
        <Title level={2} className="text-center mb-6">计数器示例</Title>
        
        <div className="flex flex-col items-center">
          <div className="mb-6">
            <Text strong className="text-xl">
              当前计数: 
            </Text>
            <div className="text-4xl font-bold text-blue-600 my-2">{count}</div>
          </div>
          
          <Space size="middle">
            <Button 
              type="primary" 
              icon={<PlusOutlined />} 
              onClick={increment}
              className="bg-blue-500 hover:bg-blue-600"
            >
              增加
            </Button>
            <Button 
              type="default" 
              icon={<MinusOutlined />} 
              onClick={decrement}
            >
              减少
            </Button>
            <Button 
              type="ghost" 
              icon={<ReloadOutlined />} 
              onClick={reset}
            >
              重置
            </Button>
          </Space>
        </div>
      </Card>
      
      <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card title="特性 1" className="shadow-md">
          <p>使用 Ant Design 组件</p>
        </Card>
        <Card title="特性 2" className="shadow-md">
          <p>集成 Zustand 状态管理</p>
        </Card>
        <Card title="特性 3" className="shadow-md">
          <p>支持 Tailwind CSS 样式</p>
        </Card>
      </div>
    </MainLayout>
  );
}
EOF

# 创建全局样式文件
cat > src/app/globals.css << EOF
@tailwind base;
@tailwind components;
@tailwind utilities;

:root {
  --foreground-rgb: 0, 0, 0;
  --background-start-rgb: 214, 219, 220;
  --background-end-rgb: 255, 255, 255;
}

@media (prefers-color-scheme: dark) {
  :root {
    --foreground-rgb: 255, 255, 255;
    --background-start-rgb: 0, 0, 0;
    --background-end-rgb: 0, 0, 0;
  }
}

body {
  color: rgb(var(--foreground-rgb));
  background: linear-gradient(
      to bottom,
      transparent,
      transparent
    )
    rgb(var(--background-start-rgb));
}

@layer utilities {
  .text-balance {
    text-wrap: balance;
  }
}

/* 自定义 Ant Design 样式覆盖 */
.ant-card {
  border-radius: 0.5rem;
}

.ant-btn {
  border-radius: 0.375rem;
}
EOF

# 创建自定义_app.tsx来整合Ant Design主题
cat > src/app/layout.tsx << EOF
import './globals.css';
import type { Metadata } from 'next';
import { Inter } from 'next/font/google';
import { ConfigProvider } from 'antd';
import zhCN from 'antd/locale/zh_CN';
import 'antd/dist/reset.css';

const inter = Inter({ subsets: ['latin'] });

export const metadata: Metadata = {
  title: 'Create Next App',
  description: 'Generated by create next app',
};

export default function RootLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  return (
    <html lang="en">
      <body className={inter.className}>
        <ConfigProvider locale={zhCN}>
          {children}
        </ConfigProvider>
      </body>
    </html>
  );
}
EOF

echo "项目 $PROJECT_NAME 已成功创建在 $OUTPUT_DIR/$PROJECT_NAME"
echo "请运行以下命令启动项目:"
echo "cd $OUTPUT_DIR/$PROJECT_NAME"
echo "npm run dev"