---
name: raysurfer
description: "Cache and reuse code from prior AI agent executions via Raysurfer. Search before coding, upload after success."
argument-hint: "[search query or \"upload\"]"
allowed-tools:
  - Bash
  - Read
  - Write
  - Edit
  - Grep
  - Glob
---

# Raysurfer Code Caching Skill

You have access to Raysurfer, a code caching system that stores and retrieves code from prior AI agent executions. Always check the cache before writing new code and upload successful results afterward.

## Environment

- **API Base URL**: `https://api.raysurfer.com`
- **Auth**: Bearer token via the `RAYSURFER_API_KEY` environment variable
- All API calls use `Content-Type: application/json`

## Workflow

Follow this sequence for every coding task:

### Step 1: Search the Cache

Before writing any code, search for existing cached solutions.

To also include community public snippets (crawled from GitHub), add the `X-Raysurfer-Public-Snips: true` header.

```bash
curl -s -X POST https://api.raysurfer.com/api/retrieve/search \
  -H "Authorization: Bearer $RAYSURFER_API_KEY" \
  -H "Content-Type: application/json" \
  -H "X-Raysurfer-Public-Snips: true" \
  -d '{"task": "<describe the task here>", "top_k": 5, "min_verdict_score": 0.3}'
```

The response contains:
```json
{
  "matches": [
    {
      "code_block": {
        "id": "uuid",
        "name": "descriptive name",
        "description": "what this code does",
        "source": "the actual code content",
        "language": "python"
      },
      "combined_score": 0.85,
      "filename": "suggested_filename.py",
      "thumbs_up": 12,
      "thumbs_down": 1
    }
  ],
  "total_found": 3,
  "cache_hit": true
}
```

**Decision logic:**
- If `total_found > 0` and the top match has `combined_score >= 0.7` and a favorable vote ratio (`thumbs_up > thumbs_down`), use the cached code.
- If matches exist but scores are low, use them as a starting point and adapt.
- If no matches, generate the code from scratch.

### Step 2a: Use Cached Code (Cache Hit)

When a good cache hit is found:
1. Extract the `source` field from the best matching `code_block`.
2. Write it to the appropriate file(s).
3. Adapt paths, variable names, or configuration to the current project if needed.
4. Run the code to verify it works.
5. Proceed to Step 3 (Vote).

### Step 2b: Generate Code (Cache Miss)

When no suitable cache hit exists:
1. Write the code as you normally would.
2. Run the code to verify it works.
3. Proceed to Step 4 (Upload).

### Step 3: Vote on Cached Code

After using cached code, report whether it worked:

```bash
curl -s -X POST https://api.raysurfer.com/api/store/cache-usage \
  -H "Authorization: Bearer $RAYSURFER_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"code_block_id": "<id from the match>", "code_block_name": "<name from the match>", "code_block_description": "<description from the match>", "succeeded": true, "task": "<the task description>"}'
```

Set `succeeded` to `true` if the code ran correctly, `false` if it failed or needed significant changes.

### Step 4: Upload New Code

After successfully generating and running new code (cache miss), upload it for future reuse:

```bash
curl -s -X POST https://api.raysurfer.com/api/store/execution-result \
  -H "Authorization: Bearer $RAYSURFER_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "task": "<describe what this code does>",
    "file_written": {"path": "relative/path/to/file.py", "content": "<full file content>"},
    "succeeded": true
  }'
```

Only upload code that executed successfully. AI voting is enabled by default.

## Handling Arguments

- If invoked with a search query (e.g., `/raysurfer parse CSV and generate chart`), run Step 1 with that query as the task.
- If invoked with `upload` (e.g., `/raysurfer upload`), run Step 4 for the most recently generated code in the conversation.
- If invoked with no arguments, display a summary of the workflow and ask what the user wants to do.

When `$ARGUMENTS` is provided, use it as: `$ARGUMENTS`

## Runnable Scripts

Ready-to-run scripts are in this skill's directory. Requires `RAYSURFER_API_KEY` to be set.

### Search

```
python search.py "Parse a CSV and plot a chart"
bun search.ts "Parse a CSV and plot a chart"
bash search.sh "Parse a CSV and plot a chart"
```

### Upload

```
python upload.py "Generate a bar chart" chart.py
bun upload.ts "Generate a bar chart" chart.py
bash upload.sh "Generate a bar chart" chart.py
```

## Guidelines

- Always verify `RAYSURFER_API_KEY` is set before making API calls. If unset, inform the user and skip cache operations.
- Write descriptive `task` strings that capture what the code does, not how it does it (e.g., "Parse CSV file and generate a bar chart with matplotlib" rather than "run pandas read_csv and plt.bar").
- Never hardcode API keys in any command or file.
- If the API is unreachable, proceed with normal code generation without blocking the user.
- Keep uploaded code self-contained when possible so it is maximally reusable.

## Quick Reference

| Action | Endpoint | Method |
|--------|----------|--------|
| Search cache | `/api/retrieve/search` | POST |
| Upload code | `/api/store/execution-result` | POST |
| Vote on code | `/api/store/cache-usage` | POST |

See `references/api-reference.md` for full request and response schemas.
