#!/usr/bin/env bash
# OpenClaw Ralph Mode Loop Script
# Simplified Ralph Wiggum adaptation for OpenClaw's sessions_spawn
# Usage: ./loop.sh [max_iterations]

set -euo pipefail

# Parse arguments
MAX_ITERATIONS=${1:-0}
PLAN_FILE="IMPLEMENTATION_PLAN.md"
AGENTS_FILE="AGENTS.md"
SPECS_DIR="specs"

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "OpenClaw Ralph Mode"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "Max iterations: ${MAX_ITERATIONS:-unlimited}"
echo "Plan: $PLAN_FILE"
echo "Agents: $AGENTS_FILE"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

# Check for required files
if [ ! -f "$PLAN_FILE" ]; then
  echo "Error: $PLAN_FILE not found. Create it first with planning phase."
  exit 1
fi

if [ ! -f "$AGENTS_FILE" ]; then
  echo "Warning: $AGENTS_FILE not found. Creating default..."
  echo "# Project Operations" > "$AGENTS_FILE"
  echo "" >> "$AGENTS_FILE"
  echo "## Build Commands" >> "$AGENTS_FILE"
  echo "npm run dev    # Development" >> "$AGENTS_FILE"
  echo "npm run build  # Production build" >> "$AGENTS_FILE"
  echo "" >> "$AGENTS_FILE"
  echo "## Validation" >> "$AGENTS_FILE"
  echo "npm run test      # All tests" >> "$AGENTS_FILE"
  echo "npm run typecheck  # TypeScript" >> "$AGENTS_FILE"
  echo "npm run lint      # ESLint" >> "$AGENTS_FILE"
fi

ITERATION=0

# Main loop
while true; do
  # Check iteration limit
  if [ $MAX_ITERATIONS -gt 0 ] && [ $ITERATION -ge $MAX_ITERATIONS ]; then
    echo ""
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo "Reached max iterations: $MAX_ITERATIONS"
    echo "To continue, re-run without iteration limit"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    break
  fi

  ITERATION=$((ITERATION + 1))
  echo ""
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
  echo "ITERATION $ITERATION"
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

  # Read current plan and check completion
  if ! grep -q "^- \[ \]" "$PLAN_FILE"; then
    echo "✅ All tasks completed!"
    echo ""
    echo "Summary: All items in $PLAN_FILE marked as done"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    break
  fi

  # Show next uncompleted task
  echo ""
  echo "Next task (uncompleted):"
  grep "^- \[ \]" "$PLAN_FILE" | head -1
  echo ""

  # Ask user to continue
  read -p "Continue with next task? [Y/n] " -n 1 -r
  if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Paused. Resume by running ./loop.sh again"
    exit 0
  fi

  echo ""
  echo "📋 Instructions for sub-agent:"
  echo "─────────────────────────────────────────────────"
  echo "1. Study AGENTS.md for build/test commands"
  echo "2. Read IMPLEMENTATION_PLAN.md for context"
  echo "3. Pick one uncompleted task to implement"
  echo "4. Implement the task (code changes only)"
  echo "5. Run validation (tests, lint, typecheck)"
  echo "6. If validation passes:"
  echo "   - Mark task as [x] in IMPLEMENTATION_PLAN.md"
  echo "   - Commit changes with descriptive message"
  echo "   - Update plan with any discoveries"
  echo ""
  echo "⚠️  Backpressure Gates:"
  echo "   - If tests FAIL: Fix before marking complete"
  echo "   - If typecheck FAILS: Fix types before committing"
  echo "   - If lint FAILS: Fix style issues before committing"
  echo "   - Only commit when ALL gates pass"
  echo ""
  read -p "Sub-agent ready? [Press Enter to spawn] " -r

  echo ""
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
  echo "Spawning sub-agent..."
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
  echo ""

  # Note: The actual sub-agent spawn happens in OpenClaw
  # This script is a guide/coordination wrapper
  # The main OpenClaw agent will handle sessions_spawn

  echo "After sub-agent completes:"
  echo "1. Run validation commands from AGENTS.md"
  echo "2. Check IMPLEMENTATION_PLAN.md for updates"
  echo "3. Run this script again for next iteration"
  echo ""

done

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "Ralph Mode complete"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
