# Code Patterns Reference

Patterns that sub-agents discover and learn from. Consistent patterns reduce iteration time and improve quality.

## Discovery

Sub-agents learn patterns during Phase 0c of the loop:

> "Study src/lib/* with up to 250 parallel Sonnet subagents to understand shared utilities & components"

When sub-agents find useful patterns, document them here.

## Common Patterns

### Database Access

**Pattern:** Use centralized db utilities from src/lib/db/

❌ Don't:
```typescript
// In multiple files scattered across src/
const pool = mysql.createPool(/* config */);
const connection = await pool.getConnection();
```

✅ Do:
```typescript
// In src/lib/db.ts
export const pool = mysql.createPool(/* config */);
export async function query<T>(sql: string, params?: any[]): Promise<T[]> {
  const conn = await pool.getConnection();
  try {
    const [results] = await conn.query(sql, params);
    return results as T[];
  } finally {
    conn.release();
  }
}

// Use in implementation
import { query } from '@/lib/db';
const users = await query<User[]>('SELECT * FROM users');
```

**Benefits:**
- Single source of truth for database config
- Reusable connection management
- Type-safe queries
- Easier to test (mock one function)

### Error Handling

**Pattern:** Consistent error types and handling

❌ Don't:
```typescript
// Scattered error approaches
if (error) return { error: "Failed" };
if (err) throw new Error("Bad request");
if (e) return res.status(500).json({ error: e.message });
```

✅ Do:
```typescript
// In src/lib/errors.ts
export class AppError extends Error {
  constructor(
    message: string,
    public statusCode: number = 500,
    public isOperational: boolean = false
  ) {
    super(message);
    this.name = 'AppError';
  }
}

export function handleAppError(error: unknown, res: Response) {
  if (error instanceof AppError) {
    return res.status(error.statusCode).json({ error: error.message });
  }
  return res.status(500).json({ error: 'Internal server error' });
}

// Use in routes
try {
  // code
} catch (error) {
  handleAppError(error, res);
}
```

### Authentication

**Pattern:** Centralized auth with token verification

❌ Don't:
```typescript
// JWT verification repeated everywhere
const token = req.headers.authorization?.split(' ')[1];
const decoded = jwt.verify(token, SECRET);
if (!decoded) return res.status(401);
```

✅ Do:
```typescript
// In src/lib/auth.ts
export function verifyToken(token: string): JWTPayload {
  const decoded = jwt.verify(token, SECRET) as JWTPayload;
  return decoded;
}

export function requireAuth(req: Request, res: Response, next: NextFunction) {
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) throw new AppError('No token', 401);
  const decoded = verifyToken(token);
  (req as any).user = decoded;
  next();
}

// Use in routes
router.get('/protected', requireAuth, async (req, res) => {
  const user = (req as any).user;
  // ...
});
```

### API Responses

**Pattern:** Consistent response structure

❌ Don't:
```typescript
// Inconsistent response formats
return { data: user };
return { user };
return { success: true, user: user, timestamp: Date.now() };
```

✅ Do:
```typescript
// In src/lib/api.ts
export interface ApiResponse<T> {
  data?: T;
  error?: string;
  success: boolean;
}

export function successResponse<T>(data: T): ApiResponse<T> {
  return { data, success: true };
}

export function errorResponse(message: string): ApiResponse<never> {
  return { error: message, success: false };
}

// Use in routes
return res.json(successResponse(user));
return res.status(400).json(errorResponse('Invalid input'));
```

### Next.js Specific Patterns

### Server Actions

**Pattern:** Server actions for mutations

❌ Don't:
```typescript
// Route handlers doing business logic
app.post('/api/users', async (req, res) => {
  const { name, email } = req.body;
  await db.query('INSERT INTO users...');
  return res.json({ success: true });
});
```

✅ Do:
```typescript
// Server action in src/app/actions/users.ts
'use server';

export async function createUser(prevState: any, formData: FormData) {
  const { name, email } = Object.fromEntries(formData);
  await db.query('INSERT INTO users...');
  revalidatePath('/users');
  return { success: true };
}

// Use in form
import { createUser } from '@/app/actions/users';
```

### Data Fetching

**Pattern:** Server components with async patterns

❌ Don't:
```typescript
// Client-side data fetching in server component
export default async function Page() {
  const res = await fetch('/api/data');
  const data = await res.json();
  // ...
}
```

✅ Do:
```typescript
// In src/app/data.ts (server utilities)
import { db } from '@/lib/db';

export async function getData() {
  return await db.query<Data[]>('SELECT * FROM data');
}

// In page
import { getData } from '@/app/data';
export default async function Page() {
  const data = await getData();
  // ...
}
```

### Python Patterns

### Pydantic Models

**Pattern:** Use Pydantic for request/response validation

❌ Don't:
```python
# Manual validation everywhere
@app.post("/users")
async def create_user(request):
    data = await request.json()
    if not data.get("name"):
        raise HTTPException(400, "Name required")
    if "@" not in data.get("email"):
        raise HTTPException(400, "Invalid email")
```

✅ Do:
```python
# In src/models/user.py
class UserCreate(BaseModel):
    name: str
    email: EmailStr

# In API
@app.post("/users")
async def create_user(user: UserCreate):
    # Validation automatic
    result = await db.insert(user.dict())
    return result
```

### FastAPI Dependencies

**Pattern:** Dependency injection with lifespan

❌ Don't:
```python
# Global state scattered
db = Database()

@app.on_event("startup")
async def startup():
    db.connect()

@app.on_event("shutdown")
async def shutdown():
    db.disconnect()
```

✅ Do:
```python
# In src/lib/database.py
class Database:
    def __init__(self):
        self.pool = None

    async def connect(self):
        self.pool = await create_pool()

    async def disconnect(self):
        if self.pool:
            self.pool.close()

# In main.py
db = Database()

@app.on_event("startup")
async def startup():
    await db.connect()

@app.on_event("shutdown")
async def shutdown():
    await db.disconnect()

# Use in routes
from src.lib.database import db

@app.get("/users")
async def get_users():
    return await db.query("SELECT * FROM users")
```

## Discovering New Patterns

When sub-agents find patterns that work well, add them here:

```markdown
## [Discovered Date]

### New Pattern Name

**Problem:** [What this pattern solves]
**Solution:** [How it works]
**Example:** [Code snippet]
**Benefits:** [Why this is better]
```

Sub-agents will read this file during Phase 0c and learn from documented patterns.
