# Backpressure Gates Reference

Backpressure gates are the mechanism that rejects incomplete or invalid work automatically, forcing the agent to iterate until quality standards are met.

## Core Concept

Instead of telling the agent "how to do it," engineer conditions where good outcomes emerge naturally through iteration.

## Types of Backpressure

### 1. Programmatic Gates (Deterministic)

These are hard gates that must pass before work is considered complete:

**Tests**
- All unit tests pass: `npm run test` / `pytest`
- Integration tests pass: `npm run test:e2e` / `pytest tests/integration`
- Coverage thresholds met: e.g., 80%+

**Typecheck**
- TypeScript: `npx tsc --noEmit`
- Python: `mypy src/`
- No type errors allowed

**Lint**
- ESLint: `npm run lint` / `eslint src/`
- Ruff: `ruff check src/`
- Style enforcement

**Build**
- Bundle compiles: `npm run build`
- Production artifact created
- No build errors

### 2. Subjective Gates (LLM-as-Judge)

For criteria that resist automation, use LLM reviews with binary pass/fail:

**UX Quality**
- Navigation is intuitive and discoverable
- Primary actions are obvious
- Error messages are helpful

**Design Quality**
- Visual hierarchy is clear
- Brand consistency maintained
- Layout follows patterns

**Content Quality**
- Tone is appropriate for audience
- Information hierarchy is logical
- No awkward phrasing

## Wiring Backpressure into Ralph Mode

### In AGENTS.md

```markdown
# Validation

## Programmatic Gates (Run these after implementing)
- Tests: npm run test          # All tests must pass
- Typecheck: npx tsc --noEmit # No type errors
- Lint: npm run lint             # Style must pass
- Build: npm run build            # Bundle compiles

## Subjective Gates (Use LLM-as-Judge for these)
- UX Review: Review user flow for intuitiveness
- Design Review: Check visual hierarchy and consistency
- Tone Review: Verify messaging appropriateness
```

### In Loop Instructions

When sub-agent completes work:

1. **Check programmatic gates first**
   - Run all validation commands
   - Fix until all pass

2. **Then check subjective gates** (if applicable)
   - Spawn LLM-as-judge sub-agent
   - Binary pass/fail only
   - Iterate until pass

3. **Only then mark task complete**
   - Update IMPLEMENTATION_PLAN.md with [x]
   - Commit changes

## Anti-Patterns

❌ **Don't skip gates**
   - "Tests pass locally" → no, must actually run
   - "Will fix lint later" → no, fix now
   - "Looks good" → verify with gates

❌ **Don't cheat completion**
   - Marking [x] without tests passing
   - Stub implementations that skip tests
   - Assuming "should work" without validation

✅ **Do enforce quality**
   - All gates must pass before committing
   - Backpressure drives iteration and quality
   - Natural convergence through enforcement

## Example Gate Failure Flow

```
Sub-agent: "I implemented user authentication"

Main agent: "Run validation gates"

Sub-agent runs: npm run test
Result: FAIL - 3 tests failing

Sub-agent:
  1. Read test failures
  2. Fix authentication logic
  3. Re-run tests

Sub-agent runs: npm run test
Result: PASS

Sub-agent runs: npm run typecheck
Result: PASS

Sub-agent runs: npm run lint
Result: FAIL - unused variable

Sub-agent:
  1. Remove unused variable
  2. Re-run lint

Sub-agent runs: npm run lint
Result: PASS

Main agent: "All gates pass. Mark task [x] in plan"

Sub-agent: Updates IMPLEMENTATION_PLAN.md with [x] User authentication
Sub-agent: git commit -m "feat: implement user authentication with JWT"

Next iteration...
```

## When Gates Don't Exist

If project has no tests yet, create them first:

1. **Start with test gates:**
   - Write tests for new feature
   - Fail initially (expected)
   - Implement to make them pass

2. **Add other gates incrementally:**
   - Once tests pass, add typecheck
   - Then add lint
   - Finally add build validation

3. **Document gate additions in AGENTS.md**
   - Keep operational guide up to date
   - Future iterations discover new gates automatically
