#!/usr/bin/env node
/**
 * Ralph Loop Runner for Clawdbot
 * 
 * Continuously runs an AI agent on a task until completion.
 * Named after the Ralph Wiggum technique by Geoffrey Huntley.
 * 
 * Uses `clawdbot agent` CLI to inject messages into a session.
 * Each iteration is a fresh context that builds on saved file state.
 * 
 * Usage:
 *   node ralph-loop.mjs --prompt "path/to/PROMPT.md" --done "DONE" --max 50
 *   node ralph-loop.mjs --prompt "Build feature X" --done "COMPLETE" --max 30
 * 
 * Options:
 *   --prompt, -p    Path to prompt file OR inline prompt text
 *   --done, -d      Completion signal to look for (default: "RALPH_DONE")
 *   --max, -m       Maximum iterations (default: 50, safety limit)
 *   --delay         Seconds between iterations (default: 5)
 *   --state         Path to state file (default: /tmp/ralph-state.json)
 *   --session       Session ID to use (default: generates new one)
 *   --thinking      Thinking level: off|minimal|low|medium|high (default: off)
 *   --timeout       Timeout per iteration in seconds (default: 300)
 *   --quiet, -q     Less output
 */

import { readFileSync, writeFileSync, existsSync, mkdirSync, unlinkSync, renameSync, copyFileSync } from 'fs';
import { join, dirname, basename } from 'path';
import { spawn, execSync } from 'child_process';
import { randomUUID } from 'crypto';
import { homedir } from 'os';

// Parse args
const args = process.argv.slice(2);
function getArg(names, defaultVal) {
  for (let i = 0; i < args.length; i++) {
    if (names.includes(args[i]) && args[i + 1]) {
      return args[i + 1];
    }
  }
  return defaultVal;
}
function hasFlag(names) {
  return args.some(a => names.includes(a));
}

const promptArg = getArg(['--prompt', '-p'], null);
const doneSignal = getArg(['--done', '-d'], 'RALPH_DONE');
const maxIterations = parseInt(getArg(['--max', '-m'], '50'));
const delaySeconds = parseInt(getArg(['--delay'], '5'));
const sessionId = getArg(['--session'], `ralph-${randomUUID().slice(0, 8)}`);
const thinking = getArg(['--thinking'], 'off');
const timeoutSecs = parseInt(getArg(['--timeout'], '300'));
const quiet = hasFlag(['--quiet', '-q']);
const fresh = hasFlag(['--fresh', '-f']); // Force fresh start, ignore existing state
const model = getArg(['--model', '-M'], null); // Model override (e.g., sonnet, opus, haiku)
const timeLimit = getArg(['--time', '-t'], null); // e.g., "1h", "30m", "23:59"
const checkCmd = getArg(['--check-cmd'], null); // Command to run before each iteration
const hardStopTimestamp = getArg(['--hard-stop'], null); // Unix timestamp for hard stop
const loopName = getArg(['--name', '-n'], null); // Human-readable name for the loop

// Generate unique state file if not specified
// Format: ralph-YYYYMMDD-HHMMSS-xxxx.json
function generateStateFilename() {
  const now = new Date();
  const ts = now.toISOString().replace(/[-:T]/g, '').slice(0, 14); // YYYYMMDDHHMMSS
  const hash = sessionId.slice(-4);
  return `/tmp/ralph-${ts}-${hash}.json`;
}

const stateFileArg = getArg(['--state'], null);
const stateFile = stateFileArg || generateStateFilename();

// Archive directory for completed runs
const archiveDir = join(homedir(), 'clawd', 'logs', 'ralph-archive');

if (!promptArg) {
  console.error('Usage: ralph-loop.mjs --prompt "task or path" [--done SIGNAL] [--max N] [--time LIMIT]');
  console.error('');
  console.error('Options:');
  console.error('  --prompt, -p    Path to prompt file OR inline prompt text');
  console.error('  --done, -d      Completion signal (default: RALPH_DONE)');
  console.error('  --max, -m       Max iterations (default: 50)');
  console.error('  --delay         Seconds between iterations (default: 5)');
  console.error('  --time, -t      Time limit: duration (1h, 30m) or end time (23:59)');
  console.error('  --thinking      Thinking level: off|minimal|low|medium|high');
  console.error('  --session       Session ID (default: random)');
  console.error('  --timeout       Seconds per iteration (default: 300)');
  console.error('  --state         State file path (default: auto-generated unique file)');
  console.error('  --fresh, -f     Force fresh start, ignore any existing state file');
  console.error('  --model, -M     Model to use (e.g., sonnet, opus, haiku). Default: claude default');
  console.error('  --quiet, -q     Less output');
  console.error('  --check-cmd     Command to run before each iteration (exit non-zero or output STOP to halt)');
  console.error('  --hard-stop     Unix timestamp for hard stop (cannot be overridden)');
  console.error('');
  console.error('Completed runs are archived to: ~/clawd/logs/ralph-archive/');
  process.exit(1);
}

// Parse time limit
function getEndTime(limit) {
  if (!limit) return null;
  
  // Check if it's a clock time (HH:MM)
  if (/^\d{1,2}:\d{2}$/.test(limit)) {
    const [hours, mins] = limit.split(':').map(Number);
    const end = new Date();
    end.setHours(hours, mins, 0, 0);
    if (end < new Date()) end.setDate(end.getDate() + 1); // Tomorrow if past
    return end;
  }
  
  // Parse duration (1h, 30m, 1h30m)
  const now = Date.now();
  let ms = 0;
  const hours = limit.match(/(\d+)h/);
  const mins = limit.match(/(\d+)m/);
  if (hours) ms += parseInt(hours[1]) * 60 * 60 * 1000;
  if (mins) ms += parseInt(mins[1]) * 60 * 1000;
  return ms > 0 ? new Date(now + ms) : null;
}

const endTime = getEndTime(timeLimit);

// Get the prompt text
function getPrompt() {
  // Check if it's a file path
  if (existsSync(promptArg)) {
    return readFileSync(promptArg, 'utf8');
  }
  // Otherwise treat as inline prompt
  return promptArg;
}

// Load/save state
function loadState() {
  // If --fresh flag, always start clean
  if (fresh && existsSync(stateFile)) {
    log('🧹 Fresh start requested, clearing existing state');
    unlinkSync(stateFile);
    const doneFile = stateFile.replace('.json', '-done.txt');
    if (existsSync(doneFile)) unlinkSync(doneFile);
  }
  
  if (existsSync(stateFile)) {
    try {
      const loaded = JSON.parse(readFileSync(stateFile, 'utf8'));
      // Ensure sessionId is always set
      if (!loaded.sessionId) loaded.sessionId = sessionId;
      log(`📂 Resuming from existing state (iteration ${loaded.iteration})`);
      return loaded;
    } catch {}
  }
  return { 
    iteration: 0, 
    started: new Date().toISOString(), 
    done: false, 
    history: [],
    sessionId,
    name: loopName, // Human-readable name
    stateFile // Track which state file is being used
  };
}

function saveState(state) {
  const dir = dirname(stateFile);
  if (!existsSync(dir)) mkdirSync(dir, { recursive: true });
  writeFileSync(stateFile, JSON.stringify(state, null, 2));
}

// Check if done signal was written
function checkDone() {
  const doneFile = stateFile.replace('.json', '-done.txt');
  if (existsSync(doneFile)) {
    const content = readFileSync(doneFile, 'utf8');
    if (content.includes(doneSignal)) {
      return true;
    }
  }
  return false;
}

// Clear the done file
function clearDone() {
  const doneFile = stateFile.replace('.json', '-done.txt');
  writeFileSync(doneFile, '');
}

// Archive completed state file
function archiveState(state) {
  try {
    // Ensure archive directory exists
    if (!existsSync(archiveDir)) {
      mkdirSync(archiveDir, { recursive: true });
    }
    
    // Generate archive filename with timestamp and status
    const status = state.done ? 'done' : (state.maxedOut ? 'maxed' : 'stopped');
    const archiveName = basename(stateFile).replace('.json', `-${status}.json`);
    const archivePath = join(archiveDir, archiveName);
    
    // Copy state file to archive
    copyFileSync(stateFile, archivePath);
    
    // Also copy done file if it exists
    const doneFile = stateFile.replace('.json', '-done.txt');
    if (existsSync(doneFile)) {
      copyFileSync(doneFile, archivePath.replace('.json', '-done.txt'));
    }
    
    log(`📦 Archived to: ${archivePath}`);
    
    // Clean up tmp files
    if (existsSync(stateFile)) unlinkSync(stateFile);
    if (existsSync(doneFile)) unlinkSync(doneFile);
    
    return archivePath;
  } catch (err) {
    console.error('⚠️ Failed to archive state:', err.message);
    return null;
  }
}

// Run agent iteration via Claude CLI (lightweight, minimal context)
async function runIteration(prompt, iteration, state) {
  const doneFile = stateFile.replace('.json', '-done.txt');
  
  const fullPrompt = `[Ralph Loop - Iteration ${iteration}/${maxIterations}]

${prompt}

---
RALPH INSTRUCTIONS:
1. Continue working on the task above
2. Save progress to files (state persists between iterations)
3. When COMPLETELY done, write "${doneSignal}" to ${doneFile}:
   echo "${doneSignal}" > "${doneFile}"
4. If not done, describe what you did and what's left
5. Be concise - no apologies, just execute`;

  return new Promise((resolve) => {
    // Use Claude CLI with minimal context (runs from workspace, reads CLAUDE.md only)
    const cliArgs = [
      '--print',
      '--dangerously-skip-permissions',
      '--output-format', 'json'
    ];
    
    // Add model override if specified
    if (model) {
      cliArgs.push('--model', model);
    }
    
    cliArgs.push(fullPrompt);
    
    const child = spawn('claude', cliArgs, {
      cwd: join(homedir(), 'clawd'),  // Run from workspace root
      stdio: ['ignore', 'pipe', 'pipe'],
      timeout: (timeoutSecs + 30) * 1000
    });
    
    let output = '';
    let error = '';
    
    child.stdout.on('data', (data) => {
      output += data.toString();
    });
    
    child.stderr.on('data', (data) => {
      error += data.toString();
    });
    
    child.on('close', (code) => {
      // Try to parse JSON output from Claude CLI
      let result = { success: code === 0, code };
      try {
        const json = JSON.parse(output);
        result.response = json.result;
        result.tokens = json.usage?.input_tokens + json.usage?.output_tokens + 
                       (json.usage?.cache_creation_input_tokens || 0) + 
                       (json.usage?.cache_read_input_tokens || 0);
        result.cost = json.total_cost_usd;
        if (!quiet && result.response) {
          console.log('\n📝 Agent response:');
          console.log(result.response.slice(0, 500) + (result.response.length > 500 ? '...' : ''));
        }
        if (!quiet && result.cost) {
          console.log(`   Cost: $${result.cost.toFixed(4)}`);
        }
      } catch {
        result.rawOutput = output;
        if (!quiet && output) {
          console.log('\n📝 Raw output:', output.slice(0, 300));
        }
      }
      
      if (error && !quiet) {
        console.log('⚠️ Stderr:', error.slice(0, 200));
      }
      
      resolve(result);
    });
    
    child.on('error', (err) => {
      resolve({
        success: false,
        error: err.message
      });
    });
  });
}

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function log(...args) {
  if (!quiet) console.log(...args);
}

// Main loop
async function main() {
  const prompt = getPrompt();
  let state = loadState();
  
  console.log('🔁 Ralph Loop Started (Claude CLI)');
  console.log(`   Prompt: ${promptArg.slice(0, 60)}${promptArg.length > 60 ? '...' : ''}`);
  console.log(`   Done signal: ${doneSignal}`);
  console.log(`   Max iterations: ${maxIterations}`);
  console.log(`   Engine: Claude CLI (minimal context)`);
  console.log(`   Model: ${model || 'default'}`);
  console.log(`   Session: ${state.sessionId}`);
  if (state.name) console.log(`   Name: ${state.name}`);
  if (endTime) console.log(`   Time limit: until ${endTime.toLocaleTimeString()}`);
  if (hardStopTimestamp) console.log(`   Hard stop: ${new Date(parseInt(hardStopTimestamp) * 1000).toLocaleTimeString()}`);
  if (checkCmd) console.log(`   Check cmd: ${checkCmd}`);
  console.log(`   State file: ${stateFile}`);
  console.log('');
  
  // Clean up previous done file
  clearDone();
  
  // Save initial state immediately so dashboard can see the loop
  saveState(state);
  
  while (state.iteration < maxIterations) {
    // ═══ HARD STOP CHECK (cannot be overridden) ═══
    if (hardStopTimestamp && Date.now() >= parseInt(hardStopTimestamp) * 1000) {
      console.log('\n⛔ HARD STOP TIME - exiting immediately');
      state.hardStopped = true;
      state.hardStopReason = 'Hard stop timestamp reached';
      saveState(state);
      break;
    }
    
    // ═══ CHECK COMMAND (runs before agent, external control) ═══
    if (checkCmd) {
      try {
        const checkResult = execSync(checkCmd, { 
          encoding: 'utf8',
          timeout: 30000,
          stdio: ['pipe', 'pipe', 'pipe']
        });
        
        if (checkResult.includes('STOP')) {
          console.log(`\n⛔ Check command signaled stop: ${checkResult.trim()}`);
          state.checkStopped = true;
          state.checkStopReason = checkResult.trim();
          saveState(state);
          break;
        }
        
        log(`   Check: ${checkResult.trim()}`);
        
      } catch (err) {
        // Check command failed or returned non-zero - STOP for safety
        const stderr = err.stderr?.toString() || '';
        const stdout = err.stdout?.toString() || '';
        const output = stdout || stderr || err.message;
        
        if (output.includes('STOP')) {
          console.log(`\n⛔ Check command signaled stop: ${output.trim()}`);
          state.checkStopped = true;
          state.checkStopReason = output.trim();
        } else {
          console.log(`\n⛔ Check command failed (exit ${err.status}) - stopping for safety`);
          console.log(`   Output: ${output.slice(0, 200)}`);
          state.checkError = true;
          state.checkErrorReason = output.trim();
        }
        saveState(state);
        break;
      }
    }
    
    // ═══ TIME LIMIT CHECK ═══
    if (endTime && new Date() >= endTime) {
      console.log(`\n⏰ Time limit reached (${endTime.toLocaleTimeString()})`);
      state.timeUp = true;
      saveState(state);
      break;
    }
    
    const iterStart = Date.now();
    const currentIter = state.iteration + 1;  // What we're about to run
    
    // Save state with current iteration info (but don't increment completed count yet)
    state.currentIterationStarted = new Date().toISOString();
    state.runningIteration = currentIter;
    saveState(state);
    
    const timeLeft = endTime ? ` | ${Math.round((endTime - Date.now()) / 60000)}m left` : '';
    log(`\n━━━ Iteration ${currentIter}/${maxIterations}${timeLeft} ━━━`);
    
    // Run the agent
    log('🤖 Running agent...');
    const result = await runIteration(prompt, currentIter, state);
    
    if (!result.success) {
      console.error('❌ Iteration failed:', result.error || `exit code ${result.code}`);
      state.history.push({ 
        iteration: currentIter, 
        error: result.error || `exit ${result.code}`,
        timestamp: new Date().toISOString()
      });
      saveState(state);
      
      // On failure, wait longer before retry
      await sleep(Math.min(delaySeconds * 2, 30) * 1000);
      continue;
    }
    
    // Give filesystem time to sync
    await sleep(1000);
    
    // Check if done
    if (checkDone()) {
      state.iteration = currentIter;  // Record completed iteration
      state.done = true;
      state.completedAt = new Date().toISOString();
      delete state.runningIteration;
      saveState(state);
      
      console.log(`\n✅ RALPH COMPLETE after ${state.iteration} iterations!`);
      const totalTime = Math.round((Date.now() - new Date(state.started).getTime()) / 1000);
      console.log(`   Total time: ${Math.floor(totalTime / 60)}m ${totalTime % 60}s`);
      
      // Archive completed run
      archiveState(state);
      return;
    }
    
    // Iteration succeeded - now increment completed count
    state.iteration = currentIter;
    delete state.runningIteration;
    
    const durationMs = Date.now() - iterStart;
    state.history.push({
      iteration: currentIter,
      timestamp: new Date().toISOString(),
      durationMs,
      tokens: result.tokens
    });
    saveState(state);
    
    log(`   Iteration took ${Math.round(durationMs / 1000)}s`);
    
    // Delay before next iteration
    if (currentIter < maxIterations) {
      log(`   Waiting ${delaySeconds}s before next iteration...`);
      await sleep(delaySeconds * 1000);
    }
  }
  
  if (!state.done && !state.timeUp) {
    console.log(`\n⚠️ Max iterations (${maxIterations}) reached without completion`);
    state.maxedOut = true;
    saveState(state);
  }
  
  // Archive the run (completed, maxed out, or stopped)
  archiveState(state);
}

main().catch(err => {
  console.error('Fatal error:', err);
  process.exit(1);
});
