import express from 'express';
import cors from 'cors';
import path from 'path';
import { fileURLToPath } from 'url';
import loopsRouter from './routes/loops.mjs';

const __dirname = path.dirname(fileURLToPath(import.meta.url));

const app = express();
const PORT = process.env.PORT || 3939;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Logging middleware
app.use((req, res, next) => {
  const timestamp = new Date().toISOString();
  console.log(`[${timestamp}] ${req.method} ${req.url}`);
  next();
});

// API Routes
app.use('/api/loops', loopsRouter);

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    message: 'Q Dashboard API is running',
    timestamp: new Date().toISOString(),
    version: '1.0.0'
  });
});

// Stats endpoint (redirect to loops stats)
app.get('/api/stats', (req, res) => {
  res.redirect('/api/loops/stats');
});

// Serve static files from public directory
app.use(express.static(path.join(__dirname, 'public')));

// Catch-all handler for SPA routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({
    success: false,
    error: 'Internal server error'
  });
});

// Start server
app.listen(PORT, () => {
  console.log(`🎯 Q Dashboard running on http://localhost:${PORT}`);
  console.log(`📊 API endpoints:`);
  console.log(`   GET  /api/health           - Health check`);
  console.log(`   GET  /api/loops            - List all loops`);
  console.log(`   GET  /api/loops/all        - All sessions (active + historical)`);
  console.log(`   GET  /api/loops/:id        - Single loop details`);
  console.log(`   GET  /api/loops/:id/transcript - Full transcript`);
  console.log(`   POST /api/loops/:id/kill   - Kill session`);
  console.log(`   GET  /api/stats            - Dashboard statistics`);
  console.log(`📁 Static files served from: ${path.join(__dirname, 'public')}`);
  console.log(`🌐 CORS enabled for all origins`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('📡 Received SIGTERM, shutting down gracefully...');
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('\n📡 Received SIGINT, shutting down gracefully...');
  process.exit(0);
});

export default app;