import express from 'express';
import { RalphReader } from '../lib/ralph-reader.mjs';

const router = express.Router();
const ralphReader = new RalphReader();

/**
 * GET /api/loops - List all Ralph loops
 */
router.get('/', async (req, res) => {
  try {
    const data = ralphReader.getAllLoops();
    
    res.json({
      success: true,
      data: {
        active: data.active,
        archived: data.archived.slice(0, 50), // Limit archived to last 50
        stats: data.stats,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('Error getting loops:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/loops/active - List only active/running loops
 */
router.get('/active', async (req, res) => {
  try {
    const active = ralphReader.getActiveLoops();
    
    res.json({
      success: true,
      data: {
        loops: active,
        count: active.length,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('Error getting active loops:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/loops/archived - List archived loops
 */
router.get('/archived', async (req, res) => {
  try {
    const limit = parseInt(req.query.limit) || 50;
    const archived = ralphReader.getArchivedLoops().slice(0, limit);
    
    res.json({
      success: true,
      data: {
        loops: archived,
        count: archived.length,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('Error getting archived loops:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/loops/stats - Dashboard statistics
 */
router.get('/stats', async (req, res) => {
  try {
    const data = ralphReader.getAllLoops();
    
    res.json({
      success: true,
      data: {
        ...data.stats,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('Error getting stats:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/loops/:id - Get single loop details
 */
router.get('/:id', async (req, res) => {
  try {
    const loop = ralphReader.getLoop(req.params.id);
    
    if (!loop) {
      return res.status(404).json({
        success: false,
        error: 'Loop not found'
      });
    }
    
    res.json({
      success: true,
      data: loop
    });
  } catch (error) {
    console.error('Error getting loop:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/loops/:id/history - Get iteration history
 */
router.get('/:id/history', async (req, res) => {
  try {
    const history = ralphReader.getLoopHistory(req.params.id);
    
    if (!history) {
      return res.status(404).json({
        success: false,
        error: 'Loop not found'
      });
    }
    
    res.json({
      success: true,
      data: history
    });
  } catch (error) {
    console.error('Error getting history:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * POST /api/loops/:id/kill - Kill a running loop
 */
router.post('/:id/kill', async (req, res) => {
  try {
    const result = ralphReader.killLoop(req.params.id);
    
    if (result.success) {
      res.json({
        success: true,
        message: result.message
      });
    } else {
      res.status(400).json({
        success: false,
        error: result.error
      });
    }
  } catch (error) {
    console.error('Error killing loop:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

export default router;
