// Q Dashboard - Frontend Logic

class QDashboard {
    constructor() {
        this.autoRefreshEnabled = true;
        this.refreshInterval = 10000; // 10 seconds
        this.refreshTimer = null;
        this.currentSessionToKill = null;
        
        this.initializeElements();
        this.setupEventListeners();
        this.startAutoRefresh();
        this.loadData();
    }

    initializeElements() {
        // Stats elements
        this.statsElements = {
            totalSessions: document.getElementById('totalSessions'),
            activeLoops: document.getElementById('activeLoops'),
            severeLoops: document.getElementById('severeLoops'),
            lastUpdated: document.getElementById('lastUpdated')
        };

        // Table elements
        this.activeLoopsBody = document.getElementById('activeLoopsBody');
        this.historicalLoopsBody = document.getElementById('historicalLoopsBody');

        // Control elements
        this.refreshBtn = document.getElementById('refreshBtn');
        this.toggleAutoRefreshBtn = document.getElementById('toggleAutoRefresh');
        this.autoRefreshStatus = document.getElementById('autoRefreshStatus');

        // Modal elements
        this.transcriptModal = document.getElementById('transcriptModal');
        this.modalTitle = document.getElementById('modalTitle');
        this.modalBody = document.getElementById('modalBody');
        this.closeModal = document.getElementById('closeModal');

        // Kill modal elements
        this.killModal = document.getElementById('killModal');
        this.killSessionLabel = document.getElementById('killSessionLabel');
        this.killSessionKey = document.getElementById('killSessionKey');
        this.confirmKill = document.getElementById('confirmKill');
        this.cancelKill = document.getElementById('cancelKill');
        this.closeKillModal = document.getElementById('closeKillModal');
    }

    setupEventListeners() {
        // Control buttons
        this.refreshBtn.addEventListener('click', () => this.loadData());
        this.toggleAutoRefreshBtn.addEventListener('click', () => this.toggleAutoRefresh());

        // Modal controls
        this.closeModal.addEventListener('click', () => this.hideTranscriptModal());
        this.closeKillModal.addEventListener('click', () => this.hideKillModal());
        this.cancelKill.addEventListener('click', () => this.hideKillModal());
        this.confirmKill.addEventListener('click', () => this.executeKill());

        // Close modals on background click
        this.transcriptModal.addEventListener('click', (e) => {
            if (e.target === this.transcriptModal) {
                this.hideTranscriptModal();
            }
        });

        this.killModal.addEventListener('click', (e) => {
            if (e.target === this.killModal) {
                this.hideKillModal();
            }
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.hideTranscriptModal();
                this.hideKillModal();
            }
            if (e.key === 'r' && (e.ctrlKey || e.metaKey)) {
                e.preventDefault();
                this.loadData();
            }
        });
    }

    async loadData() {
        try {
            this.setLoading(true);
            
            // Load both active loops and stats
            const [loopsResponse, statsResponse] = await Promise.all([
                fetch('/api/loops/all'),
                fetch('/api/stats')
            ]);

            if (!loopsResponse.ok || !statsResponse.ok) {
                throw new Error('Failed to fetch data');
            }

            const loopsData = await loopsResponse.json();
            const statsData = await statsResponse.json();

            if (loopsData.success && statsData.success) {
                this.updateStats(statsData.data);
                this.updateTables(loopsData.data.sessions);
                this.updateLastUpdated();
            } else {
                throw new Error('API returned error');
            }

        } catch (error) {
            console.error('Error loading data:', error);
            this.showError('Failed to load data. Check console for details.');
        } finally {
            this.setLoading(false);
        }
    }

    updateStats(stats) {
        this.statsElements.totalSessions.textContent = stats.totalSessions || 0;
        this.statsElements.activeLoops.textContent = stats.loopingSessions || 0;
        this.statsElements.severeLoops.textContent = stats.loopsByPattern?.severe || 0;
    }

    updateTables(sessions) {
        const activeSessions = sessions.filter(s => s.isActive);
        const historicalSessions = sessions.filter(s => !s.isActive);

        this.renderSessionTable(this.activeLoopsBody, activeSessions, true);
        this.renderSessionTable(this.historicalLoopsBody, historicalSessions, false);
    }

    renderSessionTable(tbody, sessions, isActive) {
        if (sessions.length === 0) {
            tbody.innerHTML = `
                <tr class="empty-state">
                    <td colspan="7">${isActive ? 'No active sessions' : 'No historical sessions'}</td>
                </tr>
            `;
            return;
        }

        // Sort sessions - loops first, then by start time
        sessions.sort((a, b) => {
            if (a.loopAnalysis?.isLoop && !b.loopAnalysis?.isLoop) return -1;
            if (!a.loopAnalysis?.isLoop && b.loopAnalysis?.isLoop) return 1;
            return new Date(b.startTime || 0) - new Date(a.startTime || 0);
        });

        tbody.innerHTML = sessions.map(session => this.renderSessionRow(session, isActive)).join('');

        // Add click handlers for transcript viewing
        tbody.querySelectorAll('.clickable').forEach((row, index) => {
            row.addEventListener('click', () => this.showTranscript(sessions[index]));
        });

        // Add click handlers for kill buttons
        tbody.querySelectorAll('.kill-btn').forEach((btn, index) => {
            btn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.showKillModal(sessions[index]);
            });
        });
    }

    renderSessionRow(session, isActive) {
        const status = this.getSessionStatus(session);
        const pattern = session.loopAnalysis?.pattern || 'single';
        const duration = this.formatDuration(session.startTime, session.endTime);
        const sessionKeyShort = this.truncateSessionKey(session.key);

        return `
            <tr class="${session.hasTranscript ? 'clickable' : ''}">
                <td><span class="status-badge ${status.class}">${status.text}</span></td>
                <td>${this.escapeHtml(session.label || 'Unlabeled')}</td>
                <td><span class="session-key" title="${this.escapeHtml(session.key)}">${sessionKeyShort}</span></td>
                <td>${session.loopAnalysis?.loopCount || 1}</td>
                <td><span class="pattern-badge pattern-${pattern}">${this.formatPattern(pattern)}</span></td>
                <td>${duration}</td>
                <td>
                    <div class="action-buttons">
                        ${session.hasTranscript ? 
                            `<button class="btn btn-secondary btn-small" onclick="event.stopPropagation();">📋 View</button>` : 
                            ''
                        }
                        ${isActive ? 
                            `<button class="btn btn-danger btn-small kill-btn">🔥 Kill</button>` : 
                            ''
                        }
                    </div>
                </td>
            </tr>
        `;
    }

    getSessionStatus(session) {
        if (session.isActive) {
            if (session.loopAnalysis?.isLoop) {
                return { class: 'status-loop', text: '🔄 Loop' };
            }
            return { class: 'status-running', text: '▶️ Running' };
        }
        return { class: 'status-complete', text: '✅ Complete' };
    }

    formatPattern(pattern) {
        const patterns = {
            'single': 'Single',
            'minor-repeat': 'Minor',
            'moderate-loop': 'Moderate',
            'severe-loop': 'Severe'
        };
        return patterns[pattern] || pattern;
    }

    formatDuration(startTime, endTime) {
        if (!startTime) return '-';
        
        const start = new Date(startTime);
        const end = endTime ? new Date(endTime) : new Date();
        const diffMs = end - start;
        
        if (diffMs < 60000) return '< 1m';
        if (diffMs < 3600000) return `${Math.floor(diffMs / 60000)}m`;
        if (diffMs < 86400000) return `${Math.floor(diffMs / 3600000)}h ${Math.floor((diffMs % 3600000) / 60000)}m`;
        return `${Math.floor(diffMs / 86400000)}d`;
    }

    truncateSessionKey(key) {
        if (!key) return '-';
        const parts = key.split(':');
        const uuid = parts[parts.length - 1];
        return uuid ? uuid.substring(0, 8) + '...' : key.substring(0, 12) + '...';
    }

    async showTranscript(session) {
        if (!session.hasTranscript) return;

        this.modalTitle.textContent = `📋 ${session.label || 'Unlabeled'} - Transcript`;
        this.modalBody.innerHTML = '<div class="loading">Loading transcript...</div>';
        this.showTranscriptModal();

        try {
            const response = await fetch(`/api/loops/${encodeURIComponent(session.key)}/transcript`);
            const data = await response.json();

            if (data.success) {
                this.renderTranscript(data.data);
            } else {
                throw new Error(data.error || 'Failed to load transcript');
            }
        } catch (error) {
            console.error('Error loading transcript:', error);
            this.modalBody.innerHTML = '<div class="error">Failed to load transcript</div>';
        }
    }

    renderTranscript(transcript) {
        const messages = transcript.messages || [];
        const thinking = transcript.thinking || [];
        
        // Combine and sort by timestamp
        const entries = [
            ...messages.map(m => ({ ...m, type: 'message' })),
            ...thinking.map(t => ({ ...t, type: 'thinking' }))
        ].sort((a, b) => new Date(a.timestamp || 0) - new Date(b.timestamp || 0));

        if (entries.length === 0) {
            this.modalBody.innerHTML = '<div class="empty-state">No transcript data available</div>';
            return;
        }

        const transcriptHtml = entries.map(entry => {
            const timestamp = this.formatTimestamp(entry.timestamp);
            const isThinking = entry.type === 'thinking';
            
            return `
                <div class="transcript-entry ${isThinking ? 'transcript-thinking' : ''}">
                    <div class="transcript-meta">
                        <span class="transcript-role">${isThinking ? '🧠 Thinking' : (entry.role || 'Unknown')}</span>
                        <span class="transcript-timestamp">${timestamp}</span>
                    </div>
                    <div class="transcript-content">${this.escapeHtml(entry.content || '')}</div>
                </div>
            `;
        }).join('');

        this.modalBody.innerHTML = `
            <div class="transcript-summary">
                <p><strong>Messages:</strong> ${transcript.messages?.length || 0}</p>
                <p><strong>Thinking Blocks:</strong> ${transcript.thinking?.length || 0}</p>
                <p><strong>Total Tokens:</strong> ${transcript.usage?.totalTokens || 0}</p>
            </div>
            <hr style="margin: 20px 0; border: 1px solid #1e293b;">
            ${transcriptHtml}
        `;
    }

    showKillModal(session) {
        this.currentSessionToKill = session;
        this.killSessionLabel.textContent = session.label || 'Unlabeled';
        this.killSessionKey.textContent = session.key;
        this.showKillModalElement();
    }

    async executeKill() {
        if (!this.currentSessionToKill) return;

        try {
            this.confirmKill.disabled = true;
            this.confirmKill.textContent = '🔄 Killing...';

            const response = await fetch(`/api/loops/${encodeURIComponent(this.currentSessionToKill.key)}/kill`, {
                method: 'POST'
            });

            const data = await response.json();

            if (data.success) {
                this.hideKillModal();
                this.loadData(); // Refresh the data
                this.showSuccess('Session killed successfully');
            } else {
                throw new Error(data.error || 'Failed to kill session');
            }
        } catch (error) {
            console.error('Error killing session:', error);
            this.showError('Failed to kill session: ' + error.message);
        } finally {
            this.confirmKill.disabled = false;
            this.confirmKill.textContent = '🔥 Kill Session';
            this.currentSessionToKill = null;
        }
    }

    // Auto-refresh functionality
    toggleAutoRefresh() {
        this.autoRefreshEnabled = !this.autoRefreshEnabled;
        
        if (this.autoRefreshEnabled) {
            this.startAutoRefresh();
            this.toggleAutoRefreshBtn.textContent = '⏸️ Pause Auto-Refresh';
            this.autoRefreshStatus.textContent = `Auto-refresh: ON (${this.refreshInterval / 1000}s)`;
        } else {
            this.stopAutoRefresh();
            this.toggleAutoRefreshBtn.textContent = '▶️ Start Auto-Refresh';
            this.autoRefreshStatus.textContent = 'Auto-refresh: OFF';
        }
    }

    startAutoRefresh() {
        if (!this.autoRefreshEnabled) return;
        
        this.stopAutoRefresh();
        this.refreshTimer = setInterval(() => {
            this.loadData();
        }, this.refreshInterval);
    }

    stopAutoRefresh() {
        if (this.refreshTimer) {
            clearInterval(this.refreshTimer);
            this.refreshTimer = null;
        }
    }

    // Modal management
    showTranscriptModal() {
        this.transcriptModal.classList.add('show');
    }

    hideTranscriptModal() {
        this.transcriptModal.classList.remove('show');
    }

    showKillModalElement() {
        this.killModal.classList.add('show');
    }

    hideKillModal() {
        this.killModal.classList.remove('show');
        this.currentSessionToKill = null;
    }

    // Utility functions
    updateLastUpdated() {
        const now = new Date();
        this.statsElements.lastUpdated.textContent = now.toLocaleTimeString();
    }

    formatTimestamp(timestamp) {
        if (!timestamp) return '-';
        return new Date(timestamp).toLocaleTimeString();
    }

    setLoading(loading) {
        this.refreshBtn.disabled = loading;
        this.refreshBtn.textContent = loading ? '🔄 Loading...' : '🔄 Refresh Now';
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    showError(message) {
        // Simple error notification - could be enhanced with a proper toast system
        console.error(message);
        alert('Error: ' + message);
    }

    showSuccess(message) {
        // Simple success notification - could be enhanced with a proper toast system
        console.log(message);
    }
}

// Initialize the dashboard when the page loads
document.addEventListener('DOMContentLoaded', () => {
    new QDashboard();
});