#!/usr/bin/env node
/**
 * Ralph-Evolver - Recursive Self-Improvement Engine
 *
 * Usage:
 *   node index.js --project <path>           # Single iteration
 *   node index.js --project <path> --loop 5  # Run 5 cycles
 *   node index.js --project <path> --spawn   # Output prompt for sessions_spawn
 */

const path = require('path');
const fs = require('fs');
const evolve = require('./evolve');

// State file path
function getStateFile(projectPath) {
  return path.join(projectPath, '.ralph', 'state.json');
}

function loadState(projectPath) {
  const stateFile = getStateFile(projectPath);
  try {
    if (fs.existsSync(stateFile)) {
      return JSON.parse(fs.readFileSync(stateFile, 'utf8'));
    }
  } catch (e) {
    console.error('[ralph-evolver] loadState error:', e.message);
    // Backup corrupted file
    if (fs.existsSync(stateFile)) {
      const backupPath = stateFile + '.corrupted.' + Date.now();
      try {
        fs.renameSync(stateFile, backupPath);
        console.error('[ralph-evolver] Corrupted state backed up to:', backupPath);
      } catch (backupErr) {
        console.error('[ralph-evolver] Failed to backup:', backupErr.message);
      }
    }
  }
  return { iteration: 0, loopRemaining: 0, loopTotal: 0 };
}

function saveState(projectPath, state) {
  const stateDir = path.join(projectPath, '.ralph');
  const stateFile = path.join(stateDir, 'state.json');
  try {
    if (!fs.existsSync(stateDir)) {
      fs.mkdirSync(stateDir, { recursive: true });
    }
    fs.writeFileSync(stateFile, JSON.stringify(state, null, 2));
  } catch (e) {
    console.error('Error saving state:', e.message);
  }
}

// Update loop state and show progress
function updateLoopProgress(projectPath, state) {
  state.iteration++;

  // Always save state so iteration count persists across runs
  if (state.loopRemaining > 0) {
    state.loopRemaining--;
    if (state.loopRemaining > 0) {
      saveState(projectPath, state);
      console.log(`📊 Progress: ${state.loopTotal - state.loopRemaining}/${state.loopTotal}, ${state.loopRemaining} remaining`);
      return false; // Not complete
    } else {
      console.log(`✅ All ${state.loopTotal} cycles complete!`);
      state.loopRemaining = 0;
      state.loopTotal = 0;
      saveState(projectPath, state);
      return true; // Complete
    }
  }

  // Non-loop mode: still save iteration count
  saveState(projectPath, state);
  return true;
}

function parseArgs(args) {
  const result = {
    projectPath: process.cwd(),
    loopCount: 0,
    isSpawn: false,
    isReset: false,
    task: null
  };

  for (let i = 0; i < args.length; i++) {
    const arg = args[i];
    if (arg === '--project' || arg === '-p') {
      const next = args[i + 1];
      if (next && !next.startsWith('-')) {
        result.projectPath = path.resolve(next);
        i++;
      } else {
        console.error('[ralph-evolver] --project requires a path argument');
      }
    } else if (!arg.startsWith('-') && fs.existsSync(arg)) {
      // Support positional argument: node index.js /path/to/project
      result.projectPath = path.resolve(arg);
    } else if (arg === '--loop' || arg === '--mad-dog') {
      const next = args[i + 1];
      if (next && /^\d+$/.test(next)) {
        result.loopCount = parseInt(next, 10);
        i++;
      } else {
        result.loopCount = 1;
      }
    } else if (arg === '--spawn') {
      result.isSpawn = true;
    } else if (arg === '--reset') {
      result.isReset = true;
    } else if (arg === '--task' || arg === '-t') {
      const next = args[i + 1];
      if (next && !next.startsWith('-')) {
        result.task = next;
        i++;
      } else {
        console.error('[ralph-evolver] --task requires a description argument');
      }
    }
  }

  return result;
}

async function main() {
  const args = process.argv.slice(2);

  // Show help
  if (args.includes('--help') || args.includes('-h')) {
    console.log(`
Ralph-Evolver - Recursive Self-Improvement Engine

Usage:
  node index.js [path] [options]

Options:
  --project, -p <path>   Target project path (default: current directory)
  --task, -t <desc>      Specific task description
  --loop [N]             Run N cycles (default: 1)
  --spawn                Output prompt for sessions_spawn
  --reset                Reset iteration state
  --help, -h             Show help

Examples:
  node index.js .                      # Current directory (positional)
  node index.js /path/to/app           # Specify path (positional)
  node index.js --project . --loop 5   # With loop
  node index.js --spawn                # Output for sessions_spawn
`);
    return;
  }

  const { projectPath, loopCount, isSpawn, isReset, task } = parseArgs(args);

  // Reset state
  if (isReset) {
    const stateDir = path.join(projectPath, '.ralph');
    if (fs.existsSync(stateDir)) {
      fs.rmSync(stateDir, { recursive: true });
      console.log(`🔄 Reset: removed ${stateDir}`);
    } else {
      console.log(`ℹ️  No state to reset`);
    }
    return;
  }

  console.log(`🚀 Starting Ralph-Evolver...`);
  console.log(`📁 Project: ${projectPath}`);

  // Load state
  let state = loadState(projectPath);

  // Initialize loop
  if (loopCount > 0 && state.loopRemaining === 0) {
    state.loopRemaining = loopCount;
    state.loopTotal = loopCount;
    saveState(projectPath, state);
    console.log(`🐕 **MAD DOG MODE ACTIVATED** - Running ${loopCount} cycles`);
  } else if (state.loopRemaining > 0) {
    console.log(`🔄 Cycle ${state.loopTotal - state.loopRemaining + 1}/${state.loopTotal}`);
  }

  try {
    // Generate evolution prompt
    const evolutionPrompt = evolve.generateEvolutionPrompt(projectPath, state, task);

    // Spawn mode: output prompt for sessions_spawn
    if (isSpawn) {
      console.log('\n=== EVOLUTION_PROMPT_START ===\n');
      console.log(evolutionPrompt);
      console.log('\n=== EVOLUTION_PROMPT_END ===\n');
      updateLoopProgress(projectPath, state);
      return;
    }

    // Normal mode: output prompt directly
    console.log('\n' + evolutionPrompt);
    const completed = updateLoopProgress(projectPath, state);
    if (!completed) {
      console.log(`\n⚠️  Use --spawn flag with sessions_spawn to continue the loop.`);
    }

  } catch (error) {
    console.error('Evolution failed:', error);
    console.log(`
*** 🧬 EVOLUTION ERROR ***

Cycle ${state.iteration + 1} failed.
**Error**: ${error.message}

**MISSION**:
1. **REPORT** this error
2. **FIX** the cause if possible
3. **RETRY** this cycle
    `);
    process.exit(1);
  }
}

if (require.main === module) {
  main();
}

module.exports = { loadState, saveState, parseArgs };
