import { describe, it, expect } from 'vitest';
import { getMutationDirective, HealthChecker } from './evolve.js';

describe('getMutationDirective', () => {
  it('returns REPAIR mode when there are errors', () => {
    const health = {
      errors: [{ type: 'typescript', message: 'Type error' }],
      buildSuccess: false
    };
    const result = getMutationDirective(health);
    expect(result.mode).toBe('REPAIR');
    expect(result.priority).toBe('critical');
  });

  it('returns DISCOVER mode when no errors', () => {
    const health = {
      errors: [],
      buildSuccess: true
    };
    const result = getMutationDirective(health);
    expect(result.mode).toBe('DISCOVER');
    expect(result.deepAnalysis).toBe(true);
  });
});

describe('HealthChecker.parseTestResults', () => {
  it('parses Jest format', () => {
    const checker = new HealthChecker('/tmp');
    const output = 'Tests: 5 passed, 2 failed';
    const result = checker.parseTestResults(output);
    expect(result.passed).toBe(5);
    expect(result.failed).toBe(2);
    expect(result.total).toBe(7);
  });

  it('parses Vitest format correctly (not Test Files)', () => {
    const checker = new HealthChecker('/tmp');
    // Real vitest output - should parse "Tests 8 passed" not "Test Files 1 passed"
    const output = `
 ✓ evolve.test.js  (8 tests) 8ms

 Test Files  1 passed (1)
      Tests  8 passed (8)
   Start at  09:28:31
   Duration  479ms`;
    const result = checker.parseTestResults(output);
    expect(result.passed).toBe(8);
    expect(result.failed).toBe(0);
    expect(result.total).toBe(8);
  });

  it('parses Mocha format', () => {
    const checker = new HealthChecker('/tmp');
    const output = '10 passing\n3 failing';
    const result = checker.parseTestResults(output);
    expect(result.passed).toBe(10);
    expect(result.failed).toBe(3);
  });

  it('returns zeros for unrecognized format', () => {
    const checker = new HealthChecker('/tmp');
    const result = checker.parseTestResults('random output');
    expect(result.passed).toBe(0);
    expect(result.failed).toBe(0);
  });
});

describe('HealthChecker.parseTSErrors', () => {
  it('parses TypeScript error format', () => {
    const checker = new HealthChecker('/tmp');
    const output = 'src/index.ts(10,5): error TS2304: Cannot find name "foo"';
    const errors = checker.parseTSErrors(output);
    expect(errors.length).toBe(1);
    expect(errors[0].file).toBe('src/index.ts');
    expect(errors[0].line).toBe(10);
  });

  it('returns generic error for unrecognized format with error keyword', () => {
    const checker = new HealthChecker('/tmp');
    const output = 'Some error occurred';
    const errors = checker.parseTSErrors(output);
    expect(errors.length).toBe(1);
    expect(errors[0].type).toBe('build');
  });
});

describe('HealthChecker.parsePytestResults', () => {
  it('parses pytest summary format', () => {
    const checker = new HealthChecker('/tmp');
    const output = '====== 5 passed, 2 failed in 0.12s ======';
    const result = checker.parsePytestResults(output);
    expect(result.passed).toBe(5);
    expect(result.failed).toBe(2);
    expect(result.total).toBe(7);
  });

  it('parses simple pytest format', () => {
    const checker = new HealthChecker('/tmp');
    const output = '10 passed, 3 failed';
    const result = checker.parsePytestResults(output);
    expect(result.passed).toBe(10);
    expect(result.failed).toBe(3);
  });

  it('returns default for empty output', () => {
    const checker = new HealthChecker('/tmp');
    const result = checker.parsePytestResults('');
    expect(result.passed).toBe(0);
    expect(result.failed).toBe(0);
  });
});

describe('ImprovementTracker.record with metrics', () => {
  it('records health metrics with improvement', async () => {
    const fs = await import('fs');
    const path = await import('path');
    const { ImprovementTracker } = await import('./evolve.js');

    // Use temp directory
    const tmpDir = '/tmp/ralph-test-' + Date.now();
    fs.mkdirSync(tmpDir, { recursive: true });

    const tracker = new ImprovementTracker(tmpDir);

    // Record with health snapshot
    tracker.record(
      { description: 'test improvement', insight: 'learned something' },
      { buildSuccess: true, testResults: { passed: 5, failed: 1 }, errors: [], warnings: [] }
    );

    const history = tracker.loadHistory();
    expect(history.length).toBe(1);
    expect(history[0].metrics.testPassed).toBe(5);
    expect(history[0].metrics.testFailed).toBe(1);
    expect(history[0].metrics.buildSuccess).toBe(true);

    // Cleanup
    fs.rmSync(tmpDir, { recursive: true });
  });
});
