/**
 * Ralph-Evolver Core - Evolution Engine
 *
 * Responsibilities:
 * 1. Collect project context (health check, project info)
 * 2. Determine mutation direction (REPAIR/DISCOVER)
 * 3. Generate evolution prompt
 */

const { execFileSync } = require('child_process');
const fs = require('fs');
const path = require('path');

// ============ CONFIG ============
const CONFIG = {
  maxIterations: 20,
  completionMarker: 'RALPH_CONVERGED'
};

// ============ IMPROVEMENT TRACKER ============
// Core innovation: Let ralph-evolver know what was done last round, avoiding repetition
class ImprovementTracker {
  constructor(projectPath) {
    this.projectPath = projectPath;
    this.historyFile = path.join(projectPath, '.ralph', 'improvements.json');
  }

  // Get git diff to understand last round's changes - limited to current project directory
  getLastChanges() {
    try {
      // Use -- . to limit diff scope to current directory
      const diff = execFileSync('git', ['diff', '--stat', 'HEAD~1', '--', '.'], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000
      });
      return diff || '(no changes)';
    } catch (e) {
      console.error('[ralph-evolver] getLastChanges:', e.message);
      return '(unable to get git diff)';
    }
  }

  // Time-travel analysis: Find "problem hotspots" that are repeatedly modified
  getHotspots() {
    try {
      // Find files modified most frequently in last 50 commits - limited to current directory
      const log = execFileSync('git', [
        'log', '--pretty=format:', '--name-only', '-50', '--', '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 10000
      });

      const files = log.split('\n').filter(f => f.trim());
      const counts = {};
      for (const file of files) {
        counts[file] = (counts[file] || 0) + 1;
      }

      // Sort and take top 5
      const hotspots = Object.entries(counts)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 5)
        .map(([file, count]) => `${file} (${count} changes)`);

      return hotspots.length > 0 ? hotspots.join('\n') : '(no hotspots)';
    } catch (e) {
      console.error('[ralph-evolver] getHotspots:', e.message);
      return '(unable to analyze git history)';
    }
  }

  // Find reverted changes (possibly problematic attempts) - limited to current directory
  getRevertedChanges() {
    try {
      const log = execFileSync('git', [
        'log', '--oneline', '--grep=revert', '-5', '--', '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000
      });
      return log.trim() || '(no reverts)';
    } catch (e) {
      console.error('[ralph-evolver] getRevertedChanges:', e.message);
      return '(unable to analyze)';
    }
  }

  // Load improvement history
  loadHistory() {
    try {
      if (fs.existsSync(this.historyFile)) {
        const content = fs.readFileSync(this.historyFile, 'utf-8');
        return JSON.parse(content);
      }
    } catch (e) {
      console.error('[ralph-evolver] loadHistory error:', e.message);
      // Backup corrupted file
      if (fs.existsSync(this.historyFile)) {
        const backupPath = this.historyFile + '.corrupted.' + Date.now();
        try {
          fs.renameSync(this.historyFile, backupPath);
          console.error('[ralph-evolver] Corrupted file backed up to:', backupPath);
        } catch (backupErr) {
          console.error('[ralph-evolver] Failed to backup corrupted file:', backupErr.message);
        }
      }
    }
    return [];
  }

  // Record improvement - now includes health metrics snapshot for measuring effect
  record(improvement, healthSnapshot = null) {
    try {
      const history = this.loadHistory();

      // Core improvement: Record health metrics so next round can compare effect
      const entry = {
        ...improvement,
        timestamp: new Date().toISOString(),
        metrics: healthSnapshot ? {
          buildSuccess: healthSnapshot.buildSuccess,
          testPassed: healthSnapshot.testResults?.passed || 0,
          testFailed: healthSnapshot.testResults?.failed || 0,
          errorCount: healthSnapshot.errors?.length || 0,
          warningCount: healthSnapshot.warnings?.length || 0
        } : null
      };

      history.push(entry);
      // Keep only last 10 entries
      const trimmed = history.slice(-10);

      const dir = path.dirname(this.historyFile);
      if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
      fs.writeFileSync(this.historyFile, JSON.stringify(trimmed, null, 2));
    } catch (e) {
      console.error('[ralph-evolver] record error:', e.message);
    }
  }

  // Analyze improvement effect trend
  getEffectTrend() {
    const history = this.loadHistory();
    if (history.length < 2) return null;

    // Compare metrics from last two records
    const recent = history.slice(-2);
    const [prev, curr] = recent;

    if (!prev.metrics || !curr.metrics) return null;

    const trend = {
      testDelta: (curr.metrics.testPassed - prev.metrics.testPassed) -
                 (curr.metrics.testFailed - prev.metrics.testFailed),
      errorDelta: prev.metrics.errorCount - curr.metrics.errorCount,
      buildImproved: !prev.metrics.buildSuccess && curr.metrics.buildSuccess
    };

    // Overall assessment: positive = improved, negative = degraded
    trend.overall = trend.testDelta + (trend.errorDelta * 2) + (trend.buildImproved ? 3 : 0);
    trend.verdict = trend.overall > 0 ? '📈 improved' : trend.overall < 0 ? '📉 degraded' : '➡️ unchanged';

    return trend;
  }

  // Generate history summary for prompt use
  // Includes insight and effect metrics, passing lessons learned to next round
  getSummary() {
    const history = this.loadHistory();
    if (history.length === 0) return '(first iteration, no history)';

    // Get effect trend
    const trend = this.getEffectTrend();
    let trendLine = '';
    if (trend) {
      trendLine = `\n**Last improvement effect**: ${trend.verdict} (testsΔ${trend.testDelta >= 0 ? '+' : ''}${trend.testDelta}, errorsΔ${trend.errorDelta >= 0 ? '+' : ''}${trend.errorDelta})\n`;
    }

    // Pattern analysis - count surface vs evolution levels
    const patternAnalysis = this.analyzePatterns(history);

    const historyLines = history.slice(-5).map((h, i) => {
      let line = `${i+1}. ${h.description || 'no description'}`;
      if (h.level) line += ` [${h.level}]`;
      if (h.insight) line += `\n   💡 ${h.insight}`;
      // Show metrics at time of improvement
      if (h.metrics) {
        line += `\n   📊 tests: ${h.metrics.testPassed} passed/${h.metrics.testFailed} failed, errors: ${h.metrics.errorCount}`;
      }
      return line;
    }).join('\n');

    // Add reflection prompt for pattern analysis
    const reflectionPrompt = history.length >= 3 ? `
**⚡ Reflection prompt**: Look at the improvements above. Are they mostly:
- (a) Bug fixes / edge cases / compatibility issues → surface-level
- (b) New capabilities / deeper analysis / better insights → evolution-level
If mostly (a), consider: what deeper issue is causing these surface problems?` : '';

    return patternAnalysis + trendLine + historyLines + reflectionPrompt;
  }

  // Analyze patterns across all improvements to surface meta-insights
  analyzePatterns(history) {
    if (history.length < 5) return '';

    // Count levels
    const levels = { surface: 0, evolution: 0, unlabeled: 0 };
    for (const h of history) {
      if (h.level === 'surface') levels.surface++;
      else if (h.level === 'evolution') levels.evolution++;
      else levels.unlabeled++;
    }

    // Find repeated words in insights (potential patterns)
    const words = {};
    for (const h of history) {
      if (h.insight) {
        const tokens = h.insight.toLowerCase().split(/\s+/);
        for (const t of tokens) {
          if (t.length > 4) words[t] = (words[t] || 0) + 1;
        }
      }
    }
    const repeatedWords = Object.entries(words)
      .filter(([_, count]) => count >= 2)
      .sort((a, b) => b[1] - a[1])
      .slice(0, 5)
      .map(([word, count]) => `${word}(${count})`);

    let analysis = `**📊 Pattern Analysis** (${history.length} total improvements):\n`;
    if (levels.surface + levels.evolution > 0) {
      const ratio = levels.evolution / (levels.surface + levels.evolution);
      analysis += `- Level ratio: ${levels.surface} surface / ${levels.evolution} evolution`;
      if (ratio < 0.2) analysis += ' ⚠️ mostly surface fixes';
      else if (ratio > 0.5) analysis += ' ✨ healthy evolution rate';
      analysis += '\n';
    }
    if (repeatedWords.length > 0) {
      analysis += `- Recurring themes: ${repeatedWords.join(', ')}\n`;
    }

    return analysis + '\n';
  }
}

// ============ RUNTIME SIGNALS ============
// Look not just at code structure, but what the code "says"
class RuntimeSignals {
  constructor(projectPath) {
    this.projectPath = projectPath;
  }

  // Extract TODO/FIXME comments - "distress signals" in the code
  getTodos() {
    try {
      const result = execFileSync('grep', [
        '-rn',
        '--include=*.js', '--include=*.ts', '--include=*.py',
        '--exclude-dir=node_modules', '--exclude-dir=dist', '--exclude-dir=.git',
        '-E', '(TODO|FIXME|HACK|XXX|BUG):?',
        '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000,
        maxBuffer: 50000
      });

      // Filter out self-referential noise (pattern definitions, comments about patterns)
      const lines = result.split('\n')
        .filter(l => l.trim())
        .filter(l => {
          // Exclude lines that are pattern definitions or meta-references
          if (l.includes("'-E'") || l.includes('grep')) return false;
          if (l.includes('TODO|FIXME') || l.includes('TODO/FIXME')) return false;
          if (l.includes('distress signals')) return false;
          // Must contain actual TODO/FIXME marker, not just discussion of it
          return /\b(TODO|FIXME|HACK|XXX|BUG)\b/.test(l);
        })
        .slice(0, 10);
      return lines.length > 0 ? lines.join('\n') : null;
    } catch (e) {
      // grep returns exit code 1 when no matches found, this is normal
      if (e.status !== 1) {
        console.error('[ralph-evolver] getTodos error:', e.message);
      }
      return null;
    }
  }

  // Recent commit messages - understand "why" changes were made - limited to current directory
  getRecentCommitMessages() {
    try {
      const log = execFileSync('git', [
        'log', '--oneline', '-10', '--', '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000
      });
      return log.trim() || null;
    } catch (e) {
      console.error('[ralph-evolver] getRecentCommitMessages error:', e.message);
      return null;
    }
  }

  // Error handling patterns in code - find fragile points
  getErrorPatterns() {
    try {
      const result = execFileSync('grep', [
        '-rn',
        '--include=*.js', '--include=*.ts',
        '--exclude-dir=node_modules', '--exclude-dir=dist', '--exclude-dir=.git',
        '-E', '(catch\\s*\\(|console\\.error|throw new Error)',
        '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000,
        maxBuffer: 50000
      });

      const lines = result.split('\n').filter(l => l.trim()).slice(0, 15);
      return lines.length > 0 ? lines.join('\n') : null;
    } catch (e) {
      if (e.status !== 1) {
        console.error('[ralph-evolver] getErrorPatterns error:', e.message);
      }
      return null;
    }
  }

  // Detect potentially unused exports (zombie code)
  // This catches the common problem: AI adds code that's never integrated
  getUnusedExports() {
    try {
      // Step 1: Find exported names from module.exports
      const exportsResult = execFileSync('grep', [
        '-rh',
        '--include=*.js',
        '--exclude-dir=node_modules', '--exclude-dir=dist', '--exclude-dir=.git',
        '-E', 'module\\.exports\\s*=|exports\\.[a-zA-Z]',
        '.'
      ], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 5000,
        maxBuffer: 50000
      });

      // Extract exported names
      const exportedNames = new Set();
      const lines = exportsResult.split('\n');
      for (const line of lines) {
        // Skip comments
        const trimmed = line.trim();
        if (trimmed.startsWith('//') || trimmed.startsWith('*') || trimmed.startsWith('/*')) continue;

        // Match: module.exports = { name1, name2 }
        const bracketMatch = line.match(/module\.exports\s*=\s*\{([^}]+)\}/);
        if (bracketMatch) {
          const names = bracketMatch[1].split(',').map(n => n.trim().split(':')[0].trim());
          names.forEach(n => { if (n && /^[a-zA-Z_]/.test(n)) exportedNames.add(n); });
        }
        // Match: exports.name =
        const dotMatch = line.match(/exports\.([a-zA-Z_][a-zA-Z0-9_]*)\s*=/);
        if (dotMatch) {
          exportedNames.add(dotMatch[1]);
        }
      }

      if (exportedNames.size === 0) return null;

      // Step 2: For each export, count references in the codebase
      const unused = [];
      for (const name of exportedNames) {
        // Skip common names that are likely used
        if (['default', 'module', 'exports', 'require'].includes(name)) continue;

        try {
          const countResult = execFileSync('grep', [
            '-r', '-l',
            '--include=*.js', '--include=*.ts',
            '--exclude-dir=node_modules', '--exclude-dir=dist', '--exclude-dir=.git',
            '-w', name,
            '.'
          ], {
            cwd: this.projectPath,
            encoding: 'utf-8',
            timeout: 3000
          });

          const files = countResult.split('\n').filter(f => f.trim());
          // If only found in 1 file (where it's defined), it might be unused
          if (files.length === 1) {
            unused.push(name);
          }
        } catch (e) {
          // grep returns 1 if no match - means definitely unused
          if (e.status === 1) {
            unused.push(name);
          }
        }
      }

      if (unused.length === 0) return null;
      return `Potentially unused exports: ${unused.join(', ')}`;
    } catch (e) {
      if (e.status !== 1) {
        console.error('[ralph-evolver] getUnusedExports error:', e.message);
      }
      return null;
    }
  }

  // Collect all signals
  collect() {
    const signals = {};

    const todos = this.getTodos();
    if (todos) signals.todos = todos;

    const commits = this.getRecentCommitMessages();
    if (commits) signals.commits = commits;

    const errors = this.getErrorPatterns();
    if (errors) signals.errorPatterns = errors;

    const unused = this.getUnusedExports();
    if (unused) signals.unusedExports = unused;

    return signals;
  }
}

// ============ HEALTH CHECKER ============
class HealthChecker {
  constructor(projectPath) {
    this.projectPath = projectPath;
    this._pkgCache = null; // Cache package.json
  }

  // Cached read of package.json
  getPackageJson() {
    if (this._pkgCache) return this._pkgCache;
    const pkgPath = path.join(this.projectPath, 'package.json');
    try {
      if (fs.existsSync(pkgPath)) {
        this._pkgCache = JSON.parse(fs.readFileSync(pkgPath, 'utf-8'));
      }
    } catch (e) {
      console.error('[ralph-evolver] getPackageJson error:', e.message);
      this._pkgCache = null;
    }
    return this._pkgCache;
  }

  check() {
    const health = {
      errors: [],
      warnings: [],
      testResults: null,
      buildSuccess: false,
      projectType: 'generic',
      // Documentation health status
      docs: {
        hasReadme: false,
        readmeLength: 0,
        hasChangelog: false,
        hasApiDocs: false,
        docFiles: [],
        issues: []
      }
    };

    // Detect project type
    const pkg = this.getPackageJson();
    const reqPath = path.join(this.projectPath, 'requirements.txt');
    const isDocsProject = this.isDocumentationProject();

    if (isDocsProject) {
      health.projectType = 'documentation';
      health.buildSuccess = true; // Documentation projects don't need build
    } else if (pkg) {
      health.projectType = 'nodejs';
      this.checkNodeJS(health, pkg);
    } else if (fs.existsSync(reqPath)) {
      health.projectType = 'python';
      this.checkPython(health);
    }

    // Check documentation for all projects
    this.checkDocumentation(health);

    return health;
  }

  // Detect if this is a documentation project
  isDocumentationProject() {
    const docIndicators = [
      'mkdocs.yml',
      'docusaurus.config.js',
      'docs/index.md',
      'book.toml', // mdbook
      '.vuepress/config.js'
    ];
    return docIndicators.some(f => fs.existsSync(path.join(this.projectPath, f)));
  }

  // Documentation health check
  checkDocumentation(health) {
    const docs = health.docs;

    // 1. Check README
    const readmePath = path.join(this.projectPath, 'README.md');
    if (fs.existsSync(readmePath)) {
      docs.hasReadme = true;
      const content = fs.readFileSync(readmePath, 'utf-8');
      docs.readmeLength = content.length;
      if (content.length < 200) {
        docs.issues.push({ type: 'readme_short', message: 'README.md is too short (< 200 chars)' });
      }
      if (!content.includes('## ') && !content.includes('# ')) {
        docs.issues.push({ type: 'readme_no_structure', message: 'README.md lacks heading structure' });
      }
    } else {
      docs.issues.push({ type: 'readme_missing', message: 'Missing README.md file' });
    }

    // 2. Check CHANGELOG
    const changelogPath = path.join(this.projectPath, 'CHANGELOG.md');
    docs.hasChangelog = fs.existsSync(changelogPath);

    // 3. Check API docs (common locations)
    const apiDocPaths = ['docs/', 'doc/', 'api/', 'API.md'];
    docs.hasApiDocs = apiDocPaths.some(p => fs.existsSync(path.join(this.projectPath, p)));

    // 4. Scan all .md files
    try {
      const files = fs.readdirSync(this.projectPath);
      docs.docFiles = files.filter(f => f.endsWith('.md'));
    } catch (e) {
      console.error('[ralph-evolver] checkDocumentation scan error:', e.message);
    }

    // 5. Check comment coverage of key files (simple detection)
    if (health.projectType === 'nodejs') {
      this.checkCodeComments(health);
    }
  }

  // Check code comment coverage
  checkCodeComments(health) {
    const srcDir = path.join(this.projectPath, 'src');
    if (!fs.existsSync(srcDir)) return;

    try {
      const files = fs.readdirSync(srcDir).filter(f => f.endsWith('.ts') || f.endsWith('.js'));
      let filesWithoutComments = 0;

      for (const file of files.slice(0, 5)) { // Only check first 5 files
        const content = fs.readFileSync(path.join(srcDir, file), 'utf-8');
        const hasJsDoc = content.includes('/**') || content.includes('//');
        if (!hasJsDoc && content.length > 500) {
          filesWithoutComments++;
        }
      }

      if (filesWithoutComments > 2) {
        health.docs.issues.push({
          type: 'code_no_comments',
          message: `${filesWithoutComments} source files lack comments`
        });
      }
    } catch (e) {
      console.error('[ralph-evolver] checkCodeComments error:', e.message);
    }
  }

  checkNodeJS(health, pkg) {
    // 1. Build check (only run if build script exists)
    if (pkg.scripts && pkg.scripts.build) {
      try {
        execFileSync('npm', ['run', 'build'], {
          cwd: this.projectPath,
          encoding: 'utf-8',
          timeout: 60000,
          stdio: ['pipe', 'pipe', 'pipe']
        });
        health.buildSuccess = true;
      } catch (e) {
        health.buildSuccess = false;
        const output = (e.stderr || e.stdout || e.message || '').toString();
        health.errors.push(...this.parseTSErrors(output));
      }
    } else {
      health.buildSuccess = true; // No build script = success
    }

    // 2. Test check
    const hasTestScript = pkg.scripts && pkg.scripts.test && !pkg.scripts.test.includes('no test specified');

    if (hasTestScript) {
      try {
        const output = execFileSync('npm', ['test'], {
          cwd: this.projectPath,
          encoding: 'utf-8',
          timeout: 120000,
          stdio: ['pipe', 'pipe', 'pipe']
        });
        health.testResults = this.parseTestResults(output);
      } catch (e) {
        const output = (e.stderr || e.stdout || e.message || '').toString();
        health.testResults = this.parseTestResults(output);
        if (health.testResults.failed > 0) {
          health.errors.push({
            type: 'test_failure',
            message: `Tests failed: ${health.testResults.failed} failures`
          });
        }
      }
    } else {
      health.testResults = { passed: 0, failed: 0, total: 0, skipped: true };
    }

    // 3. Lint check (if exists)
    if (pkg.scripts && pkg.scripts.lint) {
      try {
        execFileSync('npm', ['run', 'lint'], {
          cwd: this.projectPath,
          encoding: 'utf-8',
          timeout: 30000,
          stdio: ['pipe', 'pipe', 'pipe']
        });
      } catch (e) {
        const output = (e.stderr || e.stdout || e.message || '').toString();
        health.warnings.push(...this.parseLintWarnings(output));
      }
    }
  }

  checkPython(health) {
    // Python project check - find .py files and check syntax
    try {
      const pyFiles = fs.readdirSync(this.projectPath)
        .filter(f => f.endsWith('.py'))
        .slice(0, 10); // Only check first 10

      if (pyFiles.length === 0) {
        health.buildSuccess = true;
        return;
      }

      for (const file of pyFiles) {
        execFileSync('python', ['-m', 'py_compile', file], {
          cwd: this.projectPath,
          encoding: 'utf-8',
          timeout: 10000,
          stdio: ['pipe', 'pipe', 'pipe']
        });
      }
      health.buildSuccess = true;
    } catch (e) {
      health.buildSuccess = false;
      health.errors.push({
        type: 'python_syntax',
        message: e.stderr?.toString?.()?.slice(0, 200) || e.message?.slice(0, 200) || 'Python syntax error'
      });
    }

    // pytest - parse actual output for test counts
    try {
      const output = execFileSync('pytest', ['--tb=short', '-q'], {
        cwd: this.projectPath,
        encoding: 'utf-8',
        timeout: 120000,
        stdio: ['pipe', 'pipe', 'pipe']
      });
      health.testResults = this.parsePytestResults(output);
    } catch (e) {
      const output = (e.stdout || e.stderr || '').toString();
      health.testResults = this.parsePytestResults(output);
      if (health.testResults.failed > 0) {
        health.errors.push({
          type: 'test_failure',
          message: `pytest failed: ${health.testResults.failed} failures`
        });
      }
    }
  }

  // Parse pytest output: "5 passed, 2 failed" or "====== 5 passed, 2 failed in 0.12s ======"
  parsePytestResults(output) {
    const results = { passed: 0, failed: 0, total: 0 };

    // Match "X passed" and "X failed"
    const passedMatch = output.match(/(\d+)\s+passed/);
    const failedMatch = output.match(/(\d+)\s+failed/);

    if (passedMatch) results.passed = parseInt(passedMatch[1]);
    if (failedMatch) results.failed = parseInt(failedMatch[1]);

    results.total = results.passed + results.failed;

    // If no matches but pytest produced output (not empty), assume 1 passed
    // Empty output means pytest didn't run - return zeros
    if (results.total === 0 && output.trim() && !output.includes('error')) {
      results.passed = 1;
      results.total = 1;
    }

    return results;
  }

  parseTSErrors(output) {
    const errors = [];
    const errorRegex = /(.+\.ts)\((\d+),(\d+)\): error TS\d+: (.+)/g;
    let match;
    while ((match = errorRegex.exec(output)) !== null) {
      errors.push({
        type: 'typescript',
        file: match[1],
        line: parseInt(match[2]),
        message: match[4]
      });
    }
    // 如果没有匹配到具体错误但构建失败
    if (errors.length === 0 && output.includes('error')) {
      errors.push({
        type: 'build',
        message: output.slice(0, 300)
      });
    }
    return errors;
  }

  parseTestResults(output) {
    const results = { passed: 0, failed: 0, total: 0 };

    // Vitest format: "Tests  8 passed (8)" - check FIRST (most specific)
    // Must match "Tests" at start of word, not "Test Files"
    const vitestTests = output.match(/(?:^|\n)\s*Tests\s+(\d+)\s+passed/m);
    const vitestFailed = output.match(/(?:^|\n)\s*Tests\s+(\d+)\s+failed/m);
    if (vitestTests) {
      results.passed = parseInt(vitestTests[1]);
      if (vitestFailed) results.failed = parseInt(vitestFailed[1]);
      results.total = results.passed + results.failed;
      return results;
    }

    // Jest format: "Tests: X passed, Y failed"
    const jestMatch = output.match(/Tests:\s*(\d+)\s+passed(?:,\s*(\d+)\s+failed)?/);
    if (jestMatch) {
      results.passed = parseInt(jestMatch[1]);
      if (jestMatch[2]) results.failed = parseInt(jestMatch[2]);
      results.total = results.passed + results.failed;
      return results;
    }

    // Mocha format: "X passing" / "Y failing"
    const mochaPass = output.match(/(\d+)\s+passing/);
    const mochaFail = output.match(/(\d+)\s+failing/);
    if (mochaPass || mochaFail) {
      if (mochaPass) results.passed = parseInt(mochaPass[1]);
      if (mochaFail) results.failed = parseInt(mochaFail[1]);
      results.total = results.passed + results.failed;
      return results;
    }

    // Generic fallback (least specific)
    const genericPass = output.match(/(\d+)\s+passed/);
    const genericFail = output.match(/(\d+)\s+failed/);
    if (genericPass) results.passed = parseInt(genericPass[1]);
    if (genericFail) results.failed = parseInt(genericFail[1]);

    results.total = results.passed + results.failed;
    return results;
  }

  parseLintWarnings(output) {
    const warnings = [];
    const lines = output.split('\n');
    for (const line of lines) {
      if (line.includes('warning')) {
        warnings.push({ type: 'lint', message: line.trim().slice(0, 100) });
      }
    }
    return warnings.slice(0, 5);
  }
}

// ============ MUTATION DIRECTIVE ============
// Simple function, no need for a class
function getMutationDirective(health) {
  // Prioritize fixing code errors
  if (health.errors.length > 0) {
    return {
      mode: 'REPAIR',
      priority: 'critical',
      instruction: 'Fix errors, ensure build and tests pass.',
      focus: health.errors,
      deepAnalysis: false
    };
  }

  // Once stable: enter deep analysis mode
  return {
    mode: 'DISCOVER',
    priority: 'normal',
    instruction: null,
    focus: [],
    deepAnalysis: true
  };
}

// ============ PROMPT GENERATOR ============
class PromptGenerator {
  generate(projectPath, health, mutation, state, task) {
    const projectName = path.basename(projectPath);

    // REPAIR mode: straightforward fix instructions
    if (mutation.mode === 'REPAIR') {
      return this.generateRepairPrompt(projectPath, health, state, task);
    }

    // DISCOVER mode: guide deep analysis and innovation
    return this.generateDiscoverPrompt(projectPath, health, state, task);
  }

  generateRepairPrompt(projectPath, health, state, task) {
    return `
# Ralph-Evolver - Repair Mode

## Project: ${path.basename(projectPath)}
${task ? `## Task: ${task}` : ''}

## Error List
${this.formatErrors(health.errors)}

## Instructions
Fix the above errors, ensure \`npm run build\` and \`npm test\` pass.

After fixing, run verification, then continue iteration.
`;
  }

  generateDiscoverPrompt(projectPath, health, state, task) {
    const projectName = path.basename(projectPath);
    const projectContext = this.getProjectContext(projectPath);
    const fileStructure = this.getFileStructure(projectPath);

    // Get improvement history
    const tracker = new ImprovementTracker(projectPath);
    const lastChanges = tracker.getLastChanges();
    const historySummary = tracker.getSummary();

    // Collect runtime signals
    const signals = new RuntimeSignals(projectPath).collect();

    return `
# 🧬 Ralph-Evolver

**Philosophy: Recursion + Emergence**

You are iterating on project **${projectName}**.
${task ? `\n**Task**: ${task}\n` : ''}

---

## Context

### Project Status
- Path: ${projectPath}
- Type: ${health.projectType}
- Iteration: #${state.iteration + 1}
- Build: ${health.buildSuccess ? '✅' : '❌'}
- Tests: ${this.formatTestResults(health.testResults)}

### Last Round Changes
\`\`\`
${lastChanges}
\`\`\`

### Problem Hotspots (frequent changes = design issues)
${tracker.getHotspots()}
> 🔍 **Hypothesis prompt**: If a file is modified repeatedly, ask: Is it doing too much? Is its API awkward? Is it a "god object"?

### Improvement History
${historySummary}

${signals.commits ? `### Recent Commits (understand "why" changes were made)\n\`\`\`\n${signals.commits}\n\`\`\`\n> 🔍 **Hypothesis prompt**: If most commits are "fix" or "patch", what root cause keeps producing bugs?\n` : ''}

${signals.todos ? `### TODO/FIXME (distress signals in the code)\n\`\`\`\n${signals.todos}\n\`\`\`\n> 🔍 **Hypothesis prompt**: Are these TODOs procrastination, or signs of incomplete design?\n` : ''}

${signals.errorPatterns ? `### Error Handling Patterns (where are the fragile points)\n\`\`\`\n${signals.errorPatterns}\n\`\`\`\n> 🔍 **Hypothesis prompt**: Is error handling consistent (good) or scattered/ad-hoc (design smell)?\n` : ''}

${signals.unusedExports ? `### ⚠️ Zombie Code Detection\n${signals.unusedExports}\n> 🔍 **Hypothesis prompt**: These exports exist in only one file. Are they: (a) used internally but shouldn't be exported? (b) created but never integrated? (c) legacy code to delete?\n` : ''}

### Project Documentation
${projectContext}

### File Structure
${fileStructure}

### Core Code
${this.extractCoreCode(projectPath, health.projectType)}

---

## Your Task

**Think about this project from first principles.**

Don't execute a checklist. First understand:

1. **What is the essence of this project?** Why does it exist?
2. **What is it doing that it shouldn't?** What complexity is unnecessary?
3. **What is it missing that it should have?** What critical things are absent?
4. **If you started from scratch, how would you build it?** What's the gap between current design and that?

**Form hypotheses, then verify.** For each signal above (hotspots, commits, TODOs, errors), form a hypothesis about the root cause. Then check if the code confirms or refutes it.

**Change something that truly matters.** Not formatting fixes, not adding comments, not small optimizations. The kind of change where you'd say "this is right" after making it. Surface fixes are OK, but ask: is there a deeper issue causing this surface problem?

---

## Recursion: Self-Evolution

If you're iterating on **ralph-evolver itself**, you can modify \`evolve.js\`.

This is the core of recursion: the improver can improve itself.

**Meta-reflection** (only when evolving ralph-evolver):
- Am I about to fix a **surface problem** (bug, edge case, missing feature) or improve a **core capability** (better analysis, deeper insight, new signal source)?
- Look at the improvement history above—is there a **pattern**? Am I repeatedly fixing the same type of issue? That's a sign of a deeper design problem.
- The ultimate question: **After this change, will ralph-evolver be better at finding important problems?** If not, maybe the change isn't worth making.

---

## Record

After making changes, append to \`.ralph/improvements.json\`:
\`\`\`json
{
  "description": "what was done",
  "insight": "what was learned (next round will see this)",
  "level": "surface | evolution",
  "metrics": {
    "buildSuccess": true,
    "testPassed": 7,
    "testFailed": 0,
    "errorCount": 0,
    "warningCount": 0
  }
}
\`\`\`
**Level classification**:
- \`surface\`: Bug fix, edge case, compatibility, i18n, refactor without new capability
- \`evolution\`: New signal source, deeper analysis, better insight generation, meta-improvement

**Important**: Be honest about the level. If you're not sure, it's probably surface.

---

## Convergence

If after reviewing context, no problems worth fixing are found → converge: \`<promise>${CONFIG.completionMarker}</promise>\`

Don't iterate just for the sake of iterating.
`;
  }

  // Extract code structure, not blindly truncating from start
  // Core insight: AI needs to see structure (class names, function signatures), not random 800 chars
  extractCoreCode(projectPath, projectType) {
    const snippets = [];
    let coreFiles = this.findCoreFiles(projectPath, projectType);

    for (const file of coreFiles.slice(0, 3)) {
      try {
        const content = fs.readFileSync(path.join(projectPath, file), 'utf-8');
        const structure = this.extractStructure(content, file);
        snippets.push(`### ${file}\n${structure}`);
      } catch (e) {
        console.error(`[ralph-evolver] extractCoreCode error for ${file}:`, e.message);
      }
    }

    return snippets.length > 0 ? snippets.join('\n\n') : '(no core code files found)';
  }

  // Extract code structure: classes, functions, exports
  extractStructure(content, filename) {
    const lines = content.split('\n');
    const structures = [];
    let currentClass = null;
    let braceDepth = 0;
    let inClass = false;

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i];
      const lineNum = i + 1;

      // Track brace depth
      const openBraces = (line.match(/\{/g) || []).length;
      const closeBraces = (line.match(/\}/g) || []).length;

      // Detect class definition
      const classMatch = line.match(/^class\s+(\w+)/);
      if (classMatch) {
        currentClass = classMatch[1];
        inClass = true;
        braceDepth = 0;
        structures.push(`[L${lineNum}] class ${currentClass}`);
      }

      // Detect methods inside class
      if (inClass && currentClass) {
        const methodMatch = line.match(/^\s+(async\s+)?(\w+)\s*\([^)]*\)\s*\{/);
        if (methodMatch && !['if', 'for', 'while', 'switch', 'catch'].includes(methodMatch[2])) {
          const asyncPrefix = methodMatch[1] ? 'async ' : '';
          structures.push(`  [L${lineNum}] ${asyncPrefix}${methodMatch[2]}()`);
        }
      }

      braceDepth += openBraces - closeBraces;

      // Class ends
      if (inClass && braceDepth <= 0 && closeBraces > 0) {
        inClass = false;
        currentClass = null;
      }

      // Top-level functions (not in a class)
      if (!inClass) {
        const funcMatch = line.match(/^(async\s+)?function\s+(\w+)/);
        if (funcMatch) {
          const asyncPrefix = funcMatch[1] ? 'async ' : '';
          structures.push(`[L${lineNum}] ${asyncPrefix}function ${funcMatch[2]}()`);
        }

        // Arrow function exports
        const arrowMatch = line.match(/^const\s+(\w+)\s*=\s*(async\s+)?\([^)]*\)\s*=>/);
        if (arrowMatch) {
          const asyncPrefix = arrowMatch[2] ? 'async ' : '';
          structures.push(`[L${lineNum}] ${asyncPrefix}const ${arrowMatch[1]} = () =>`);
        }
      }

      // Only match actual module.exports assignment, not references in strings
      if (/^module\.exports\s*=/.test(line)) {
        structures.push(`[L${lineNum}] module.exports = {...}`);
      }
    }

    if (structures.length === 0) {
      const head = lines.slice(0, 15).join('\n');
      const tail = lines.length > 30 ? '\n...\n' + lines.slice(-10).join('\n') : '';
      return '```javascript\n' + head + tail + '\n```';
    }

    return '```\n' + structures.join('\n') + '\n```\n(' + lines.length + ' lines)';
  }

  // Find core files
  findCoreFiles(projectPath, projectType) {
    let coreFiles = [];

    if (projectType === 'nodejs') {
      const candidates = ['evolve.js', 'index.js', 'main.js', 'app.js', 'src/index.js'];
      for (const f of candidates) {
        if (fs.existsSync(path.join(projectPath, f))) {
          coreFiles.push(f);
        }
      }

      try {
        const srcDir = path.join(projectPath, 'src');
        if (fs.existsSync(srcDir)) {
          const files = fs.readdirSync(srcDir)
            .filter(f => f.endsWith('.js') || f.endsWith('.ts'))
            .map(f => ({ name: `src/${f}`, size: fs.statSync(path.join(srcDir, f)).size }))
            .sort((a, b) => b.size - a.size);
          if (files.length > 0 && !coreFiles.includes(files[0].name)) {
            coreFiles.push(files[0].name);
          }
        }
      } catch (e) {
        console.error('[ralph-evolver] findCoreFiles src scan error:', e.message);
      }
    } else if (projectType === 'python') {
      // Python project: check common entry points and largest files
      const candidates = ['main.py', 'app.py', '__main__.py', 'cli.py', 'core.py'];
      for (const f of candidates) {
        if (fs.existsSync(path.join(projectPath, f))) {
          coreFiles.push(f);
        }
      }

      // Also scan for largest .py files (likely core logic)
      try {
        const pyFiles = fs.readdirSync(projectPath)
          .filter(f => f.endsWith('.py') && !f.startsWith('test_') && f !== 'setup.py')
          .map(f => ({ name: f, size: fs.statSync(path.join(projectPath, f)).size }))
          .sort((a, b) => b.size - a.size)
          .slice(0, 3);
        for (const f of pyFiles) {
          if (!coreFiles.includes(f.name)) {
            coreFiles.push(f.name);
          }
        }
      } catch (e) {
        console.error('[ralph-evolver] findCoreFiles python scan error:', e.message);
      }
    }

    return coreFiles;
  }

  // Get file structure
  getFileStructure(projectPath) {
    try {
      const files = [];
      const scanDir = (dir, prefix = '') => {
        const entries = fs.readdirSync(dir, { withFileTypes: true });
        for (const entry of entries) {
          if (entry.name.startsWith('.') || entry.name === 'node_modules' || entry.name === 'dist') continue;
          const fullPath = path.join(dir, entry.name);
          if (entry.isDirectory()) {
            files.push(`${prefix}${entry.name}/`);
            if (files.length < 30) { // Limit depth
              scanDir(fullPath, prefix + '  ');
            }
          } else {
            files.push(`${prefix}${entry.name}`);
          }
          if (files.length >= 50) return; // Limit total
        }
      };
      scanDir(projectPath);
      return '```\n' + files.join('\n') + '\n```';
    } catch (e) {
      return '(unable to read file structure)';
    }
  }

  formatTestResults(testResults) {
    if (!testResults) return 'not run';
    if (testResults.skipped) return '⏭️ skipped';
    return `${testResults.passed}/${testResults.total} passed`;
  }

  formatErrors(errors) {
    if (errors.length === 0) return '';
    const errorList = errors.map((e, i) =>
      `${i+1}. [${e.type}] ${e.message}${e.file ? `\n   Location: ${e.file}:${e.line}` : ''}`
    ).join('\n');
    return `\n### Errors to Fix\n${errorList}\n`;
  }

  formatDocIssues(issues) {
    if (!issues || issues.length === 0) return '';
    const issueList = issues.map((e, i) =>
      `${i+1}. [${e.type}] ${e.message}`
    ).join('\n');
    return `\n### Documentation Issues\n${issueList}\n`;
  }

  getProjectContext(projectPath) {
    const contextFiles = ['README.md', 'CLAUDE.md', 'SKILL.md'];
    let context = '';

    for (const file of contextFiles) {
      const filePath = path.join(projectPath, file);
      if (fs.existsSync(filePath)) {
        const content = fs.readFileSync(filePath, 'utf-8');
        context += `### ${file}\n${content.slice(0, 400)}...\n\n`;
      }
    }

    const pkgPath = path.join(projectPath, 'package.json');
    if (fs.existsSync(pkgPath)) {
      try {
        const pkg = JSON.parse(fs.readFileSync(pkgPath, 'utf-8'));
        context += `### package.json
- name: ${pkg.name}
- version: ${pkg.version}
- scripts: ${Object.keys(pkg.scripts || {}).join(', ')}
`;
      } catch (e) {
        console.error('[ralph-evolver] getProjectContext package.json error:', e.message);
      }
    }

    return context || 'No project context files';
  }
}

// ============ EXPORT API ============

/**
 * Generate evolution prompt
 */
function generateEvolutionPrompt(projectPath, state, task) {
  const healthChecker = new HealthChecker(projectPath);
  const promptGenerator = new PromptGenerator();

  // 1. Health check
  const health = healthChecker.check();

  // 2. Mutation directive
  const mutation = getMutationDirective(health);

  // 3. Check convergence - only hard limit, Agent decides whether to converge
  if (state.iteration >= CONFIG.maxIterations) {
    return `# Ralph-Evolver Converged\n\nReached maximum iterations (${CONFIG.maxIterations}).\n\n<promise>${CONFIG.completionMarker}</promise>`;
  }

  // 4. Generate Prompt
  return promptGenerator.generate(projectPath, health, mutation, state, task);
}

module.exports = {
  generateEvolutionPrompt,
  getMutationDirective,
  HealthChecker,
  PromptGenerator,
  ImprovementTracker,
  CONFIG
};
