#!/usr/bin/env python3
"""Resolve Radarr defaults (profile/root) from env, with validation.

Env (optional):
- RADARR_DEFAULT_PROFILE (name or id)
- RADARR_DEFAULT_ROOT (path or id)

Output JSON:
{
  "profile": {"id": 4, "name": "HD-1080p"} | null,
  "root": {"id": 2, "path": "/movies"} | null,
  "needs": {"profile": bool, "root": bool},
  "options": {"profiles": [...], "roots": [...]}
}

If defaults are missing, includes options for prompting.
"""

from __future__ import annotations

import json
import os
import subprocess


def _run_json(cmd: list[str]) -> object:
    out = subprocess.check_output(cmd, stderr=subprocess.STDOUT, text=True)
    return json.loads(out)


def _radarr(args: list[str]) -> object:
    return _run_json(["bash", "-lc", "cd /home/vishix/.openclaw/workspace && ./skills/radarr/scripts/radarr.sh " + " ".join(args)])


def main() -> int:
    prof = os.environ.get("RADARR_DEFAULT_PROFILE")
    root = os.environ.get("RADARR_DEFAULT_ROOT")

    profiles = _radarr(["profiles"])
    roots = _radarr(["roots"])

    prof_obj = None
    if prof:
        # by id
        try:
            pid = int(prof)
            for p in profiles:
                if int(p.get("id")) == pid:
                    prof_obj = {"id": p.get("id"), "name": p.get("name")}
                    break
        except ValueError:
            needle = prof.strip().lower()
            for p in profiles:
                if (p.get("name") or "").strip().lower() == needle:
                    prof_obj = {"id": p.get("id"), "name": p.get("name")}
                    break

    root_obj = None
    if root:
        # by id
        try:
            rid = int(root)
            for r in roots:
                if int(r.get("id")) == rid:
                    root_obj = {"id": r.get("id"), "path": r.get("path")}
                    break
        except ValueError:
            needle = root.rstrip("/")
            for r in roots:
                if (r.get("path") or "").rstrip("/") == needle:
                    root_obj = {"id": r.get("id"), "path": r.get("path")}
                    break

    out = {
        "profile": prof_obj,
        "root": root_obj,
        "needs": {"profile": prof_obj is None, "root": root_obj is None},
        "options": {
            "profiles": [{"id": p.get("id"), "name": p.get("name")} for p in profiles],
            "roots": [{"id": r.get("id"), "path": r.get("path")} for r in roots],
        },
    }

    print(json.dumps(out, ensure_ascii=False, indent=2))
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
