#!/usr/bin/env python3
"""Track Radarr download/import progress by polling.

This is the v1 approach (no Radarr webhooks required).

Env:
- RADARR_URL
- RADARR_API_KEY

Usage:
  ./skills/radarr/scripts/radarr_track.py --movie-id 123 --interval 20 --timeout 7200

Behavior:
- Polls Radarr queue and emits JSON "events" on state changes.
- Terminates when the movie is imported (best-effort) or timeout.

Output:
- JSON lines (one per event) for easy consumption.
"""

from __future__ import annotations

import argparse
import json
import os
import time
import urllib.parse
import urllib.request


def _env(name: str) -> str:
    v = os.environ.get(name)
    if not v:
        raise SystemExit(f"Missing env var {name}. Set it in ~/.openclaw/.env")
    return v


def _base_url() -> str:
    return _env("RADARR_URL").rstrip("/")


def _api_key() -> str:
    return _env("RADARR_API_KEY")


def _request(path: str, *, params: dict | None = None):
    url = _base_url() + path
    if params:
        url += "?" + urllib.parse.urlencode(params)

    req = urllib.request.Request(
        url,
        headers={"X-Api-Key": _api_key(), "Accept": "application/json"},
        method="GET",
    )
    with urllib.request.urlopen(req, timeout=30) as resp:
        raw = resp.read()
    return json.loads(raw.decode("utf-8")) if raw else None


def _emit(evt: dict):
    print(json.dumps(evt, ensure_ascii=False))


def _queue_for_movie(movie_id: int):
    data = _request("/api/v3/queue", params={"page": 1, "pageSize": 200})
    records = (data or {}).get("records") if isinstance(data, dict) else data
    if not records:
        return []
    out = []
    for r in records:
        mid = r.get("movieId")
        if mid == movie_id:
            out.append(r)
    return out


def _movie_details(movie_id: int):
    return _request(f"/api/v3/movie/{movie_id}")


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="radarr_track")
    ap.add_argument("--movie-id", type=int, required=True)
    ap.add_argument("--interval", type=int, default=20)
    ap.add_argument("--timeout", type=int, default=7200)
    args = ap.parse_args(argv)

    start = time.time()
    last_state = None

    movie = _movie_details(args.movie_id)
    _emit({"event": "start", "movieId": args.movie_id, "title": movie.get("title"), "year": movie.get("year")})

    while True:
        if time.time() - start > args.timeout:
            _emit({"event": "timeout", "movieId": args.movie_id})
            return 2

        # Heuristic states:
        # - If queue has entry: downloading (progress available)
        # - If queue empty and movie has file: imported
        q = _queue_for_movie(args.movie_id)

        state = None
        progress = None
        if q:
            # Prefer the first record.
            r = q[0]
            state = (r.get("status") or "downloading").lower()
            sizeleft = r.get("sizeleft")
            size = r.get("size")
            if isinstance(sizeleft, (int, float)) and isinstance(size, (int, float)) and size > 0:
                progress = max(0.0, min(1.0, (size - sizeleft) / size))
        else:
            movie = _movie_details(args.movie_id)
            if movie.get("hasFile"):
                state = "imported"
            else:
                state = "pending"

        if state != last_state:
            _emit({"event": "state", "movieId": args.movie_id, "state": state, "progress": progress})
            last_state = state

        if state == "imported":
            _emit({"event": "done", "movieId": args.movie_id})
            return 0

        time.sleep(max(5, args.interval))


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
