#!/usr/bin/env python3
"""Single-shot Radarr status for a movie.

Env:
- RADARR_URL
- RADARR_API_KEY

Usage:
  ./skills/radarr/scripts/radarr_status.py --movie-id 123

Output JSON:
{
  "movieId": 123,
  "title": "...",
  "year": 2021,
  "state": "pending|downloading|imported",
  "progress": 0.0..1.0|null,
  "queue": { ... } | null
}
"""

from __future__ import annotations

import argparse
import json
import os
import urllib.parse
import urllib.request


def _env(name: str) -> str:
    v = os.environ.get(name)
    if not v:
        raise SystemExit(f"Missing env var {name}. Set it in ~/.openclaw/.env")
    return v


def _base_url() -> str:
    return _env("RADARR_URL").rstrip("/")


def _api_key() -> str:
    return _env("RADARR_API_KEY")


def _get(path: str, params: dict | None = None):
    url = _base_url() + path
    if params:
        url += "?" + urllib.parse.urlencode(params)
    req = urllib.request.Request(url, headers={"X-Api-Key": _api_key(), "Accept": "application/json"})
    with urllib.request.urlopen(req, timeout=30) as resp:
        raw = resp.read()
    return json.loads(raw.decode("utf-8")) if raw else None


def _queue_record(movie_id: int):
    data = _get("/api/v3/queue", params={"page": 1, "pageSize": 200})
    records = (data or {}).get("records") if isinstance(data, dict) else data
    if not records:
        return None
    for r in records:
        if r.get("movieId") == movie_id:
            return r
    return None


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="radarr_status")
    ap.add_argument("--movie-id", type=int, required=True)
    args = ap.parse_args(argv)

    movie = _get(f"/api/v3/movie/{args.movie_id}") or {}
    q = _queue_record(args.movie_id)

    state = "pending"
    progress = None

    if q:
        state = (q.get("status") or "downloading").lower()
        sizeleft = q.get("sizeleft")
        size = q.get("size")
        if isinstance(sizeleft, (int, float)) and isinstance(size, (int, float)) and size > 0:
            progress = max(0.0, min(1.0, (size - sizeleft) / size))
    elif movie.get("hasFile"):
        state = "imported"

    out = {
        "movieId": args.movie_id,
        "title": movie.get("title"),
        "year": movie.get("year"),
        "state": state,
        "progress": progress,
        "queue": q,
    }
    print(json.dumps(out, ensure_ascii=False))
    return 0


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
