#!/usr/bin/env python3
"""Interactive movie request helper.

This script is designed to be called by the assistant during chat.
It does NOT send messages; it outputs structured JSON that the assistant can turn into chat prompts.

Inputs:
- --query "<title>" (optional: include year)
- or --tmdb <id>

Behavior:
- Uses Radarr lookup to find candidates (works with Radarr only).
- If TMDB_API_KEY/OMDB_API_KEY exist, it can build a richer card via movie_card.py.
- Resolves defaults if RADARR_DEFAULT_PROFILE / RADARR_DEFAULT_ROOT are set.
- Otherwise returns the available profile/root options so the assistant can prompt the user.

Output JSON:
{
  "intent": "request",
  "candidates": [...],
  "needs": {"profile": bool, "root": bool, "selection": bool},
  "options": {"profiles": [...], "roots": [...]},
  "selected": { ... } | null,
  "card": { ... } | null
}

"""

from __future__ import annotations

import argparse
import json
import os
import re
import subprocess


def _run_json(cmd: list[str]) -> object:
    out = subprocess.check_output(cmd, stderr=subprocess.STDOUT, text=True)
    out = out.strip()
    return json.loads(out) if out else None


def _radarr(cmd_args: list[str]) -> object:
    # Call radarr.py directly to avoid shell-quoting issues.
    return _run_json(["python3", "/home/vishix/.openclaw/workspace/skills/radarr/scripts/radarr.py", *cmd_args])


def _env(name: str) -> str | None:
    v = os.environ.get(name)
    return v if v else None


def _parse_query(q: str) -> tuple[str, int | None]:
    # Extract trailing year like "Dune 2021" or "Dune (2021)"
    m = re.search(r"\(?\b(19\d{2}|20\d{2})\b\)?\s*$", q.strip())
    if m:
        year = int(m.group(1))
        title = q[: m.start()].strip()
        if title:
            return title, year
    return q.strip(), None


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="radarr_request")
    ap.add_argument("--query", help="Movie title, optionally with year")
    ap.add_argument("--tmdb", type=int, help="TMDB id")
    args = ap.parse_args(argv)

    if not args.query and not args.tmdb:
        raise SystemExit("Provide --query or --tmdb")

    # candidates
    if args.tmdb:
        candidates = _radarr(["lookup", "--compact", f"tmdb:{args.tmdb}"])
    else:
        title, year = _parse_query(args.query)
        candidates = _radarr(["lookup", "--compact", title])
        # keep year hint in selected logic later

    candidates = candidates or []

    selection_needed = len(candidates) != 1
    selected = candidates[0] if len(candidates) == 1 else None

    # profile/root defaults
    default_profile = _env("RADARR_DEFAULT_PROFILE")
    default_root = _env("RADARR_DEFAULT_ROOT")

    needs_profile = default_profile is None
    needs_root = default_root is None

    profiles = None
    roots = None

    if needs_profile:
        profiles = _radarr(["profiles"])
    if needs_root:
        roots = _radarr(["roots"])

    # optional card (only if tmdb add-in available AND selected has tmdbId)
    card = None
    if selected and selected.get("tmdbId") and (os.environ.get("TMDB_API_KEY") or os.environ.get("OMDB_API_KEY")):
        cmd = [
            "python3",
            "/home/vishix/.openclaw/workspace/skills/radarr/scripts/movie_card.py",
            "--tmdb",
            str(selected["tmdbId"]),
        ]
        try:
            card = _run_json(cmd)
        except Exception:
            card = None

    out = {
        "intent": "request",
        "candidates": candidates,
        "needs": {"selection": selection_needed, "profile": needs_profile, "root": needs_root},
        "options": {
            "profiles": [{"id": p.get("id"), "name": p.get("name")} for p in (profiles or [])] if profiles else None,
            "roots": [{"id": r.get("id"), "path": r.get("path")} for r in (roots or [])] if roots else None,
        },
        "selected": selected,
        "card": card,
        "defaults": {"profile": default_profile, "root": default_root},
    }

    print(json.dumps(out, ensure_ascii=False, indent=2))
    return 0


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
