#!/usr/bin/env python3
"""Radarr helper CLI.

Uses env vars:
- RADARR_URL (e.g. http://10.0.0.2:7878)
- RADARR_API_KEY

Supports:
- search/lookup
- list profiles/root folders
- add a movie by tmdbId or by lookup term

Designed to be called from OpenClaw skills via exec.
"""

from __future__ import annotations

import argparse
import json
import os
import sys
import urllib.parse
import urllib.request


def _env(name: str) -> str:
    v = os.environ.get(name)
    if not v:
        raise SystemExit(f"Missing env var {name}. Set it in ~/.openclaw/.env")
    return v


def _base_url() -> str:
    url = _env("RADARR_URL").rstrip("/")
    return url


def _api_key() -> str:
    return _env("RADARR_API_KEY")


def _request(path: str, *, method: str = "GET", params: dict | None = None, body: dict | None = None):
    base = _base_url()
    url = base + path
    if params:
        qs = urllib.parse.urlencode(params)
        url = url + ("?" if "?" not in url else "&") + qs

    headers = {
        "X-Api-Key": _api_key(),
        "Accept": "application/json",
    }

    data = None
    if body is not None:
        data = json.dumps(body).encode("utf-8")
        headers["Content-Type"] = "application/json"

    req = urllib.request.Request(url, method=method, headers=headers, data=data)
    try:
        with urllib.request.urlopen(req, timeout=30) as resp:
            raw = resp.read()
            if not raw:
                return None
            return json.loads(raw.decode("utf-8"))
    except urllib.error.HTTPError as e:
        raw = e.read().decode("utf-8", errors="replace")
        raise SystemExit(f"Radarr API error {e.code} {e.reason} for {method} {url}\n{raw}")


def _print(obj):
    print(json.dumps(obj, indent=2, ensure_ascii=False))


def cmd_ping(_args):
    # Radarr doesn't have a canonical /ping across all versions; /api/v3/system/status is reliable.
    _print(_request("/api/v3/system/status"))


def cmd_profiles(_args):
    _print(_request("/api/v3/qualityprofile"))


def cmd_roots(_args):
    _print(_request("/api/v3/rootfolder"))


def cmd_lookup(args):
    term = args.term
    # Radarr expects term like "tmdb:603" or "inception".
    res = _request("/api/v3/movie/lookup", params={"term": term})
    # Keep output compact by default
    if args.compact:
        out = []
        for m in res or []:
            out.append(
                {
                    "title": m.get("title"),
                    "year": m.get("year"),
                    "tmdbId": m.get("tmdbId"),
                    "imdbId": m.get("imdbId"),
                    "titleSlug": m.get("titleSlug"),
                }
            )
        _print(out)
        return
    _print(res)


def _find_profile_id(name_or_id: str | None) -> int | None:
    if not name_or_id:
        return None
    try:
        return int(name_or_id)
    except ValueError:
        pass

    profiles = _request("/api/v3/qualityprofile") or []
    needle = name_or_id.strip().lower()
    for p in profiles:
        if (p.get("name") or "").strip().lower() == needle:
            return int(p["id"])
    raise SystemExit(
        f"Quality profile not found: {name_or_id}. Use `profiles` to list available profiles."
    )


def _find_root_path(path_or_id: str | None) -> str | None:
    if not path_or_id:
        return None

    roots = _request("/api/v3/rootfolder") or []

    # If numeric, treat as id
    try:
        root_id = int(path_or_id)
        for r in roots:
            if int(r.get("id")) == root_id:
                return r.get("path")
        raise SystemExit(f"Root folder id not found: {root_id}. Use `roots` to list.")
    except ValueError:
        pass

    # Otherwise treat as a path
    pnorm = path_or_id.rstrip("/")
    for r in roots:
        if (r.get("path") or "").rstrip("/") == pnorm:
            return r.get("path")

    raise SystemExit(
        f"Root folder path not found in Radarr: {path_or_id}. Use `roots` to list configured root folders."
    )


def cmd_add(args):
    if not args.tmdb and not args.term:
        raise SystemExit("Provide --tmdb <id> or --term <lookup term>")

    profile_id = _find_profile_id(args.profile)
    root_path = _find_root_path(args.root)

    if args.tmdb:
        term = f"tmdb:{args.tmdb}"
        matches = _request("/api/v3/movie/lookup", params={"term": term}) or []
    else:
        matches = _request("/api/v3/movie/lookup", params={"term": args.term}) or []

    if not matches:
        raise SystemExit("No matches found.")

    # If multiple matches and no explicit tmdb, try to choose best by year if provided.
    selected = None
    if args.tmdb:
        for m in matches:
            if int(m.get("tmdbId") or 0) == int(args.tmdb):
                selected = m
                break
    if selected is None and args.year is not None:
        for m in matches:
            if m.get("year") == args.year:
                selected = m
                break
    if selected is None:
        selected = matches[0]

    # Enrich payload per Radarr requirements
    movie = dict(selected)

    # Required-ish fields for POST /movie
    movie["qualityProfileId"] = profile_id if profile_id is not None else movie.get("qualityProfileId")
    movie["rootFolderPath"] = root_path if root_path is not None else movie.get("rootFolderPath")
    movie["monitored"] = bool(args.monitor)

    if not movie.get("qualityProfileId"):
        raise SystemExit("Missing qualityProfileId. Provide --profile or set a default in Radarr.")
    if not movie.get("rootFolderPath"):
        raise SystemExit("Missing rootFolderPath. Provide --root or set a default in Radarr.")

    # Radarr expects addOptions in the POST body.
    movie["addOptions"] = {
        "monitor": "movieOnly",
        "searchForMovie": bool(args.search),
    }

    created = _request("/api/v3/movie", method="POST", body=movie)

    # Optionally trigger a search command as well (in case Radarr ignores addOptions on some setups)
    if args.search and created and created.get("id"):
        _request(
            "/api/v3/command",
            method="POST",
            body={"name": "MoviesSearch", "movieIds": [created["id"]]},
        )

    _print(
        {
            "added": True,
            "title": created.get("title") if created else movie.get("title"),
            "year": created.get("year") if created else movie.get("year"),
            "tmdbId": created.get("tmdbId") if created else movie.get("tmdbId"),
            "id": created.get("id") if created else None,
            "monitored": created.get("monitored") if created else movie.get("monitored"),
            "rootFolderPath": created.get("rootFolderPath") if created else movie.get("rootFolderPath"),
            "qualityProfileId": created.get("qualityProfileId") if created else movie.get("qualityProfileId"),
        }
    )


def main(argv: list[str]) -> int:
    p = argparse.ArgumentParser(prog="radarr")
    sub = p.add_subparsers(dest="cmd", required=True)

    sp = sub.add_parser("ping", help="Show Radarr system status")
    sp.set_defaults(func=cmd_ping)

    sp = sub.add_parser("profiles", help="List quality profiles")
    sp.set_defaults(func=cmd_profiles)

    sp = sub.add_parser("roots", help="List root folders")
    sp.set_defaults(func=cmd_roots)

    sp = sub.add_parser("lookup", help="Lookup movies")
    sp.add_argument("term", help='Lookup term, e.g. "inception" or "tmdb:603"')
    sp.add_argument("--compact", action="store_true", help="Print compact list")
    sp.set_defaults(func=cmd_lookup)

    sp = sub.add_parser("add", help="Add a movie")
    sp.add_argument("--tmdb", type=int, help="TMDB id")
    sp.add_argument("--term", help="Lookup term (title)")
    sp.add_argument("--year", type=int, help="Prefer match with this year")
    sp.add_argument("--profile", help="Quality profile name or id")
    sp.add_argument("--root", help="Root folder path or id")
    sp.add_argument("--monitor", action="store_true", help="Monitor movie")
    sp.add_argument("--search", action="store_true", help="Start search after adding")
    sp.set_defaults(func=cmd_add)

    args = p.parse_args(argv)
    args.func(args)
    return 0


if __name__ == "__main__":
    raise SystemExit(main(sys.argv[1:]))
