#!/usr/bin/env python3
"""Poll tracked Radarr movies and queue outbound notifications.

Design:
- Reads: workspace/state/radarr/tracks/*.json
- Writes: workspace/state/radarr/outbox/*.json

Outbox item schema:
{
  "channel": "telegram",
  "target": "-100..." | "384...",
  "text": "...",
  "attachment": null | "/path/to/file"
}

This script *updates* track files when it queues a notification.
It avoids spamming by only notifying on state changes.

Optional add-in:
- If PLEX_URL + PLEX_TOKEN are set, will attempt to generate a Plex link when imported.
"""

from __future__ import annotations

import json
import os
import pathlib
import subprocess
import time
import uuid

WORKSPACE = pathlib.Path(os.environ.get("OPENCLAW_WORKSPACE", os.getcwd()))
STATE_DIR = WORKSPACE / "state" / "radarr"
TRACKS_DIR = STATE_DIR / "tracks"
OUTBOX_DIR = STATE_DIR / "outbox"


def _load_json(path: pathlib.Path) -> dict:
    return json.loads(path.read_text(encoding="utf-8"))


def _write_json(path: pathlib.Path, obj: dict):
    path.write_text(json.dumps(obj, ensure_ascii=False, indent=2) + "\n", encoding="utf-8")


def _queue_message(channel: str, target: str, text: str, attachment: str | None = None):
    OUTBOX_DIR.mkdir(parents=True, exist_ok=True)
    item = {
        "id": str(uuid.uuid4()),
        "ts": int(time.time()),
        "channel": channel,
        "target": str(target),
        "text": text,
        "attachment": attachment,
    }
    path = OUTBOX_DIR / f"{item['ts']}-{item['id']}.json"
    _write_json(path, item)


def _run_json(cmd: list[str]) -> dict | None:
    try:
        out = subprocess.check_output(cmd, stderr=subprocess.STDOUT, text=True)
    except subprocess.CalledProcessError as e:
        return {"error": True, "output": e.output}

    out = out.strip()
    if not out:
        return None
    return json.loads(out)


def _plex_link(title: str | None, year: int | None) -> str | None:
    if not os.environ.get("PLEX_URL") or not os.environ.get("PLEX_TOKEN"):
        return None
    if not title:
        return None
    cmd = [
        "python3",
        str(WORKSPACE / "skills" / "radarr" / "scripts" / "plex_link.py"),
        "--title",
        title,
    ]
    if year is not None:
        cmd += ["--year", str(year)]

    res = _run_json(cmd)
    if not res or not res.get("found"):
        return None
    return res.get("url")


def main() -> int:
    if not TRACKS_DIR.exists():
        return 0

    for path in sorted(TRACKS_DIR.glob("*.json")):
        track = _load_json(path)
        if track.get("done"):
            continue

        movie_id = int(track["movieId"])
        channel = track.get("channel") or "telegram"
        target = str(track.get("target"))

        # Poll status
        status_cmd = [
            "python3",
            str(WORKSPACE / "skills" / "radarr" / "scripts" / "radarr_status.py"),
            "--movie-id",
            str(movie_id),
        ]
        status = _run_json(status_cmd)
        if not status or status.get("error"):
            # avoid loops; just skip
            continue

        state = status.get("state")
        progress = status.get("progress")
        title = status.get("title") or track.get("title")
        year = status.get("year") or track.get("year")

        last_state = track.get("lastState")

        if state != last_state:
            # Compose notification text
            label = f"{title} ({year})" if title and year else (title or f"movieId {movie_id}")
            if state == "pending":
                text = f"⏳ Radarr: queued request for {label}."
            elif state == "downloading" or state == "queued":
                pct = f" ({int(progress*100)}%)" if isinstance(progress, (int, float)) else ""
                text = f"⬇️ Radarr: downloading {label}{pct}."
            elif state == "imported":
                plex = _plex_link(title, year)
                if plex:
                    text = f"✅ Radarr: downloaded & imported {label}.\n▶️ Plex: {plex}"
                else:
                    text = f"✅ Radarr: downloaded & imported {label}."
            else:
                text = f"ℹ️ Radarr: {label} state: {state}."

            _queue_message(channel, target, text)

            track["lastState"] = state
            track["title"] = title
            track["year"] = year
            if state == "imported":
                track["done"] = True

            _write_json(path, track)

    return 0


if __name__ == "__main__":
    raise SystemExit(main())
