#!/usr/bin/env python3
"""Find a movie in Plex and produce a web URL.

Env (optional add-in):
- PLEX_URL (e.g. http://plex:32400)
- PLEX_TOKEN

Usage:
  ./skills/radarr/scripts/plex_link.py --title "Dune" --year 2021

Outputs JSON with either:
- {"found": true, "url": "...", "ratingKey": "..."}
- {"found": false, "url": "<fallback search url>"}
"""

from __future__ import annotations

import argparse
import json
import os
import urllib.parse
import urllib.request
import xml.etree.ElementTree as ET


def _env(name: str) -> str | None:
    v = os.environ.get(name)
    return v if v else None


def _plex_base() -> str | None:
    url = _env("PLEX_URL")
    return url.rstrip("/") if url else None


def _token() -> str | None:
    return _env("PLEX_TOKEN")


def _get_xml(path: str, params: dict | None = None) -> ET.Element | None:
    base = _plex_base()
    token = _token()
    if not base or not token:
        return None

    url = base + path
    q = dict(params or {})
    q["X-Plex-Token"] = token
    url = url + "?" + urllib.parse.urlencode(q)

    req = urllib.request.Request(url, headers={"Accept": "application/xml"})
    with urllib.request.urlopen(req, timeout=30) as resp:
        raw = resp.read().decode("utf-8", errors="replace")
    return ET.fromstring(raw)


def _machine_id() -> str | None:
    root = _get_xml("/")
    if root is None:
        return None
    return root.attrib.get("machineIdentifier")


def _fallback_search_url(title: str) -> str | None:
    base = _plex_base()
    if not base:
        return None
    # This opens Plex Web, then user can search.
    return f"{base}/web/index.html#!/search?query={urllib.parse.quote(title)}"


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="plex_link")
    ap.add_argument("--title", required=True)
    ap.add_argument("--year", type=int)
    args = ap.parse_args(argv)

    base = _plex_base()
    token = _token()
    if not base or not token:
        print(json.dumps({"found": False, "url": None, "reason": "PLEX_URL/PLEX_TOKEN not set"}))
        return 0

    machine = _machine_id()

    root = _get_xml("/search", params={"query": args.title})
    if root is None:
        print(json.dumps({"found": False, "url": _fallback_search_url(args.title)}))
        return 0

    best = None
    for el in root.findall(".//Video"):
        if (el.attrib.get("type") or "").lower() != "movie":
            continue
        if args.year is not None:
            try:
                if int(el.attrib.get("year") or 0) != args.year:
                    continue
            except ValueError:
                pass
        best = el
        break

    if not best:
        print(json.dumps({"found": False, "url": _fallback_search_url(args.title)}))
        return 0

    rating_key = best.attrib.get("ratingKey")
    if not rating_key:
        print(json.dumps({"found": False, "url": _fallback_search_url(args.title)}))
        return 0

    # A stable Plex web details URL (works in browsers).
    # key must be URL-encoded; ratingKey lives under /library/metadata/<id>
    details_key = f"/library/metadata/{rating_key}"
    if machine:
        url = (
            f"{base}/web/index.html#!/server/{machine}/details?key="
            + urllib.parse.quote(details_key, safe="")
        )
    else:
        url = _fallback_search_url(args.title)

    print(json.dumps({"found": True, "url": url, "ratingKey": rating_key}, ensure_ascii=False))
    return 0


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
