#!/usr/bin/env python3
"""Build a rich movie card (poster/trailer/rating) with graceful degradation.

This script is intentionally dependency-light (stdlib only).

Inputs:
- A TMDB id, OR a free-text title (+ optional year)

Data sources:
- TMDB (optional): poster + trailer + overview + TMDB rating
  env: TMDB_API_KEY
- OMDb (optional): IMDb rating
  env: OMDB_API_KEY

Output:
- JSON payload suitable for chat rendering.

Note: This script does NOT send messages; it only prepares metadata.
"""

from __future__ import annotations

import argparse
import json
import os
import re
import urllib.parse
import urllib.request

TMDB_IMAGE_BASE = "https://image.tmdb.org/t/p/"


def _get(url: str, headers: dict | None = None, timeout: int = 30, retries: int = 2):
    # TMDB sometimes resets connections; use a UA and retry a couple times.
    base_headers = {"User-Agent": "openclaw-radarr-skill"}
    if headers:
        base_headers.update(headers)

    last_err = None
    for attempt in range(retries + 1):
        try:
            req = urllib.request.Request(url, headers=base_headers)
            with urllib.request.urlopen(req, timeout=timeout) as resp:
                raw = resp.read()
            return json.loads(raw.decode("utf-8"))
        except Exception as e:
            last_err = e
            if attempt >= retries:
                raise
    raise last_err


def _tmdb_key() -> str | None:
    return os.environ.get("TMDB_API_KEY")


def _omdb_key() -> str | None:
    return os.environ.get("OMDB_API_KEY")


def tmdb_search(title: str, year: int | None = None) -> dict | None:
    key = _tmdb_key()
    if not key:
        return None
    params = {"api_key": key, "query": title}
    if year:
        params["year"] = str(year)
    url = "https://api.themoviedb.org/3/search/movie?" + urllib.parse.urlencode(params)
    data = _get(url)
    results = data.get("results") or []
    if not results:
        return None
    return results[0]


def tmdb_details(tmdb_id: int) -> dict | None:
    key = _tmdb_key()
    if not key:
        return None
    url = f"https://api.themoviedb.org/3/movie/{tmdb_id}?" + urllib.parse.urlencode(
        {"api_key": key, "append_to_response": "external_ids"}
    )
    return _get(url)


def tmdb_videos(tmdb_id: int) -> list[dict] | None:
    key = _tmdb_key()
    if not key:
        return None
    url = f"https://api.themoviedb.org/3/movie/{tmdb_id}/videos?" + urllib.parse.urlencode({"api_key": key})
    return (_get(url).get("results") or [])


def pick_trailer_url(videos: list[dict] | None) -> str | None:
    if not videos:
        return None

    def score(v: dict) -> int:
        t = (v.get("type") or "").lower()
        site = (v.get("site") or "").lower()
        name = (v.get("name") or "").lower()
        s = 0
        if site == "youtube":
            s += 10
        if t == "trailer":
            s += 10
        if "official" in name:
            s += 2
        return s

    best = max(videos, key=score)
    if (best.get("site") or "").lower() == "youtube" and best.get("key"):
        return f"https://www.youtube.com/watch?v={best['key']}"
    return None


def omdb_rating(imdb_id: str) -> dict | None:
    key = _omdb_key()
    if not key:
        return None
    params = {"apikey": key, "i": imdb_id}
    url = "https://www.omdbapi.com/?" + urllib.parse.urlencode(params)
    data = _get(url)
    if (data.get("Response") or "").lower() == "false":
        return None
    return {
        "imdbRating": data.get("imdbRating"),
        "imdbVotes": data.get("imdbVotes"),
        "rated": data.get("Rated"),
        "runtime": data.get("Runtime"),
        "genres": data.get("Genre"),
        "awards": data.get("Awards"),
    }


def _year_from_date(date_str: str | None) -> int | None:
    if not date_str:
        return None
    m = re.match(r"^(\d{4})", date_str)
    return int(m.group(1)) if m else None


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="movie_card")
    ap.add_argument("--tmdb", type=int, help="TMDB id")
    ap.add_argument("--title", help="Movie title")
    ap.add_argument("--year", type=int, help="Prefer match with this year")
    ap.add_argument("--poster-size", default="w500", help="TMDB poster size (default: w500)")

    args = ap.parse_args(argv)

    tmdb_id = args.tmdb

    # If TMDB key exists and we only have title/year, try to resolve to tmdb id.
    if not tmdb_id and args.title and _tmdb_key():
        hit = tmdb_search(args.title, args.year)
        if hit and hit.get("id"):
            tmdb_id = int(hit["id"])

    details = tmdb_details(tmdb_id) if tmdb_id else None
    videos = tmdb_videos(tmdb_id) if tmdb_id else None

    title = None
    year = None
    overview = None
    poster_url = None
    tmdb_vote = None
    imdb_id = None

    if details:
        title = details.get("title")
        year = _year_from_date(details.get("release_date"))
        overview = details.get("overview")
        tmdb_vote = details.get("vote_average")
        imdb_id = (details.get("external_ids") or {}).get("imdb_id") or details.get("imdb_id")
        poster_path = details.get("poster_path")
        if poster_path:
            poster_url = f"{TMDB_IMAGE_BASE}{args.poster_size}{poster_path}"

    # Fallback to provided title/year
    if not title and args.title:
        title = args.title
    if not year and args.year:
        year = args.year

    trailer_url = pick_trailer_url(videos)

    imdb = omdb_rating(imdb_id) if imdb_id else None

    out = {
        "title": title,
        "year": year,
        "tmdbId": tmdb_id,
        "imdbId": imdb_id,
        "overview": overview,
        "posterUrl": poster_url,
        "trailerUrl": trailer_url,
        "ratings": {
            "imdb": imdb,
            "tmdbVoteAverage": tmdb_vote,
        },
        "capabilities": {
            "tmdb": bool(_tmdb_key()),
            "omdb": bool(_omdb_key()),
        },
    }

    print(json.dumps(out, indent=2, ensure_ascii=False))
    return 0


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
