#!/usr/bin/env python3
"""Enqueue a Radarr movie tracking job.

This writes a track file under workspace/state/radarr/tracks.

Usage:
  ./skills/radarr/scripts/enqueue_track.py \
    --channel telegram \
    --target -100123... \
    --movie-id 123 \
    --title "Dune" \
    --year 2021

Output JSON: {"ok": true, "path": "..."}
"""

from __future__ import annotations

import argparse
import json
import os
import pathlib
import time


def main(argv: list[str]) -> int:
    ap = argparse.ArgumentParser(prog="enqueue_track")
    ap.add_argument("--channel", required=True)
    ap.add_argument("--target", required=True)
    ap.add_argument("--movie-id", type=int, required=True)
    ap.add_argument("--title")
    ap.add_argument("--year", type=int)
    ap.add_argument("--workspace", default=os.getcwd(), help="Workspace root (default: cwd)")
    args = ap.parse_args(argv)

    root = pathlib.Path(args.workspace)
    tracks = root / "state" / "radarr" / "tracks"
    tracks.mkdir(parents=True, exist_ok=True)

    payload = {
        "createdAt": int(time.time()),
        "channel": args.channel,
        "target": str(args.target),
        "movieId": args.movie_id,
        "title": args.title,
        "year": args.year,
        "lastState": None,
        "done": False,
    }

    path = tracks / f"{args.movie_id}.json"
    path.write_text(json.dumps(payload, ensure_ascii=False, indent=2) + "\n", encoding="utf-8")

    print(json.dumps({"ok": True, "path": str(path)}, ensure_ascii=False))
    return 0


if __name__ == "__main__":
    raise SystemExit(main(os.sys.argv[1:]))
