#!/usr/bin/env python3
"""Validate required environment variables for the radarr skill.

Usage:
  ./skills/radarr/scripts/check_env.py

Exit codes:
  0 = ok
  2 = missing required vars
"""

from __future__ import annotations

import os
import sys

REQUIRED = [
    "RADARR_URL",
    "RADARR_API_KEY",
]

OPTIONAL = [
    "RADARR_DEFAULT_PROFILE",
    "RADARR_DEFAULT_ROOT",
    "TMDB_API_KEY",
    "OMDB_API_KEY",
    "PLEX_URL",
    "PLEX_TOKEN",
]


def main() -> int:
    missing = [k for k in REQUIRED if not os.environ.get(k)]

    if missing:
        print("Missing required env vars:")
        for k in missing:
            print(f"- {k}")
        print("\nSet them in: ~/.openclaw/.env")
        print("Example:\n  RADARR_URL=http://10.0.0.2:7878\n  RADARR_API_KEY=...\n")
        return 2

    print("Radarr skill env looks OK.")
    print("Required:")
    for k in REQUIRED:
        v = os.environ.get(k, "")
        shown = v if k.endswith("URL") else ("<set>" if v else "<missing>")
        print(f"- {k}={shown}")

    print("\nOptional (for the full experience):")
    for k in OPTIONAL:
        v = os.environ.get(k)
        print(f"- {k}={'<set>' if v else '<not set>'}")

    return 0


if __name__ == "__main__":
    sys.exit(main())
