#!/usr/bin/env python3
"""
Qwen3-TTS: Local or Remote Text-to-Speech

Usage:
  tts.py [options] "Text to speak"

Options:
  -o, --output PATH        Output file path (default: qwen_output.wav)
  -s, --speaker NAME       Speaker voice (default: Vivian) - see --list-speakers
  -l, --language LANG      Language (default: Auto) - Auto/Chinese/English/Italian/etc.
  -i, --instruct TEXT      Voice instruction (emotion, style)
  --remote URL             Use remote server (e.g., http://192.168.188.177:8765)
  --list-speakers          List available speakers and exit
  --model NAME             Model name (default: Qwen/Qwen3-TTS-12Hz-0.6B-CustomVoice)
  --help                   Show this help message

Environment:
  QWEN_TTS_REMOTE          Default remote server URL
  QWEN_TTS_SPEAKER         Default speaker
  QWEN_TTS_LANGUAGE        Default language

Examples:
  tts.py "Ciao, come va?" -s Vivian -l Italian
  tts.py "Hello world" --remote http://192.168.188.177:8765
  tts.py "Sono felice!" -i "Parla con entusiasmo" -l Italian
"""

import argparse
import sys
import os
from pathlib import Path

# Find venv Python and re-exec if needed
SCRIPT_DIR = Path(__file__).parent
VENV_PYTHON = SCRIPT_DIR.parent / "venv" / "bin" / "python"

# Always use venv if available and not already active
if not os.environ.get("QWEN_TTS_VENV_ACTIVE") and VENV_PYTHON.exists():
    os.environ["QWEN_TTS_VENV_ACTIVE"] = "1"
    os.execv(str(VENV_PYTHON), [str(VENV_PYTHON)] + sys.argv)

# Check for remote mode
REMOTE_URL = os.environ.get("QWEN_TTS_REMOTE")

# Import dependencies based on mode
if REMOTE_URL or "--remote" in sys.argv:
    # Remote mode: only need requests
    try:
        import requests
    except ImportError:
        print("❌ Error: requests module required for remote mode", file=sys.stderr)
        print("   Run: venv/bin/pip install requests", file=sys.stderr)
        sys.exit(1)
else:
    # Local mode: need full TTS stack
    if not VENV_PYTHON.parent.parent.exists():
        print("❌ Error: Virtual environment not found!", file=sys.stderr)
        print(f"   Expected at: {VENV_PYTHON.parent.parent}", file=sys.stderr)
        print("", file=sys.stderr)
        print("Run setup first:", file=sys.stderr)
        print(f"   cd {SCRIPT_DIR.parent}", file=sys.stderr)
        print("   bash scripts/setup.sh", file=sys.stderr)
        sys.exit(1)
    
    try:
        import torch
        import soundfile as sf
        from qwen_tts import Qwen3TTSModel
    except ImportError as e:
        print(f"❌ Error: Missing dependency: {e}", file=sys.stderr)
        print("", file=sys.stderr)
        print("Run setup to install dependencies:", file=sys.stderr)
        print("   bash scripts/setup.sh", file=sys.stderr)
        sys.exit(1)

# Speaker information
SPEAKERS = {
    "Vivian": {"lang": "Chinese", "desc": "Bright, slightly edgy young female"},
    "Serena": {"lang": "Chinese", "desc": "Warm, gentle young female"},
    "Uncle_Fu": {"lang": "Chinese", "desc": "Seasoned male, low mellow timbre"},
    "Dylan": {"lang": "Chinese (Beijing)", "desc": "Youthful Beijing male, clear natural"},
    "Eric": {"lang": "Chinese (Sichuan)", "desc": "Lively Chengdu male, husky bright"},
    "Ryan": {"lang": "English", "desc": "Dynamic male, strong rhythmic"},
    "Aiden": {"lang": "English", "desc": "Sunny American male, clear midrange"},
    "Ono_Anna": {"lang": "Japanese", "desc": "Playful female, light nimble"},
    "Sohee": {"lang": "Korean", "desc": "Warm female, rich emotion"},
}

def list_speakers():
    """Print available speakers and their descriptions."""
    print("Available Speakers:")
    print("=" * 70)
    for name, info in SPEAKERS.items():
        print(f"  {name:12} ({info['lang']:20}) - {info['desc']}")
    print()
    print("Supported Languages:")
    print("  Auto, Chinese, English, Japanese, Korean, German, French,")
    print("  Russian, Portuguese, Spanish, Italian")


def synthesize_remote(remote_url, text, speaker, language, instruct, model, output_path):
    """Synthesize speech using remote server."""
    print(f"🌐 Using remote server: {remote_url}", file=sys.stderr)
    
    try:
        response = requests.post(
            f"{remote_url}/tts",
            json={
                "text": text,
                "speaker": speaker,
                "language": language,
                "instruct": instruct,
                "model": model
            },
            timeout=120
        )
        
        if response.status_code != 200:
            print(f"❌ Server error: {response.status_code}", file=sys.stderr)
            print(response.text, file=sys.stderr)
            sys.exit(1)
        
        # Save audio
        output_path = Path(output_path).expanduser().resolve()
        output_path.parent.mkdir(parents=True, exist_ok=True)
        
        with open(output_path, 'wb') as f:
            f.write(response.content)
        
        print(f"✅ Audio saved: {output_path}", file=sys.stderr)
        print(str(output_path))
        
    except requests.exceptions.RequestException as e:
        print(f"❌ Connection error: {e}", file=sys.stderr)
        print(f"   Is server running at {remote_url}?", file=sys.stderr)
        sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description="Qwen3-TTS: Local or remote text-to-speech",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog=__doc__
    )
    
    parser.add_argument("text", nargs="?", help="Text to synthesize")
    parser.add_argument("-o", "--output", default="qwen_output.wav",
                       help="Output audio file path")
    parser.add_argument("-s", "--speaker", 
                       default=os.environ.get("QWEN_TTS_SPEAKER", "Vivian"),
                       help="Speaker voice (default: Vivian)")
    parser.add_argument("-l", "--language",
                       default=os.environ.get("QWEN_TTS_LANGUAGE", "Auto"),
                       help="Language (default: Auto)")
    parser.add_argument("-i", "--instruct", default="",
                       help="Voice instruction (emotion, style)")
    parser.add_argument("--remote",
                       default=os.environ.get("QWEN_TTS_REMOTE"),
                       help="Remote server URL (e.g., http://192.168.188.177:8765)")
    parser.add_argument("--list-speakers", action="store_true",
                       help="List available speakers")
    parser.add_argument("--model", default="Qwen/Qwen3-TTS-12Hz-0.6B-CustomVoice",
                       help="Model name")
    
    args = parser.parse_args()
    
    # Handle --list-speakers
    if args.list_speakers:
        list_speakers()
        sys.exit(0)
    
    # Require text
    if not args.text:
        parser.error("text argument required (or use --list-speakers)")
    
    # Validate speaker
    if args.speaker not in SPEAKERS:
        print(f"❌ Error: Unknown speaker '{args.speaker}'", file=sys.stderr)
        print(f"   Available: {', '.join(SPEAKERS.keys())}", file=sys.stderr)
        print("   Use --list-speakers for details", file=sys.stderr)
        sys.exit(1)
    
    # Remote mode
    if args.remote:
        synthesize_remote(
            remote_url=args.remote,
            text=args.text,
            speaker=args.speaker,
            language=args.language,
            instruct=args.instruct,
            model=args.model,
            output_path=args.output
        )
        return
    
    # Local mode - Load model
    print(f"🔄 Loading {args.model}...", file=sys.stderr)
    print(f"   (First run downloads ~1.7GB)", file=sys.stderr)
    
    try:
        # Determine device and dtype
        device = "cuda:0" if torch.cuda.is_available() else "cpu"
        dtype = torch.bfloat16 if device != "cpu" else torch.float32
        
        # Flash attention if available (CUDA only)
        attn_impl = "flash_attention_2" if device != "cpu" else "eager"
        
        model = Qwen3TTSModel.from_pretrained(
            args.model,
            device_map=device,
            dtype=dtype,
            attn_implementation=attn_impl,
        )
        print(f"✓ Model loaded on {device}", file=sys.stderr)
    except Exception as e:
        print(f"❌ Error loading model: {e}", file=sys.stderr)
        sys.exit(1)
    
    # Generate speech
    print(f"🎙️  Generating speech...", file=sys.stderr)
    print(f"   Text: {args.text[:60]}{'...' if len(args.text) > 60 else ''}", file=sys.stderr)
    print(f"   Speaker: {args.speaker} ({SPEAKERS[args.speaker]['lang']})", file=sys.stderr)
    if args.instruct:
        print(f"   Instruction: {args.instruct}", file=sys.stderr)
    
    try:
        wavs, sr = model.generate_custom_voice(
            text=args.text,
            language=args.language,
            speaker=args.speaker,
            instruct=args.instruct if args.instruct else None,
        )
        
        # Save output
        output_path = Path(args.output).expanduser().resolve()
        output_path.parent.mkdir(parents=True, exist_ok=True)
        
        sf.write(str(output_path), wavs[0], sr)
        print(f"✅ Audio saved: {output_path}", file=sys.stderr)
        
        # Print path to stdout (for OpenClaw integration)
        print(str(output_path))
        
    except Exception as e:
        print(f"❌ Error during synthesis: {e}", file=sys.stderr)
        import traceback
        traceback.print_exc(file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
