#!/usr/bin/env python3
"""
Qwen3-TTS VoiceDesign Client

Usage:
  tts-voicedesign.py [options] "Text to speak"

Options:
  -o, --output PATH        Output file path (default: qwen_output.wav)
  -v, --voice TEXT         Voice description (default: warm gentle Italian female)
  -l, --language LANG      Language (default: Italian)
  -i, --instruct TEXT      Additional instructions (e.g., "speak slowly")
  --remote URL             Remote server URL (required)
  --help                   Show this help message

Environment:
  QWEN_TTS_REMOTE          Default remote server URL

Examples:
  tts-voicedesign.py "Ciao Pasquale!" --remote http://192.168.188.177:8765
  tts-voicedesign.py "Hello world" -v "deep authoritative male voice" -l English
  tts-voicedesign.py "Buongiorno" -i "speak with enthusiasm" --remote http://192.168.188.177:8765
"""

import argparse
import sys
import os
from pathlib import Path

try:
    import requests
except ImportError:
    print("❌ Error: requests module required", file=sys.stderr)
    print("   pip install requests", file=sys.stderr)
    sys.exit(1)


DEFAULT_VOICE = "A warm, gentle young Italian female voice with clear pronunciation"


def synthesize(remote_url, text, voice_desc, language, instruct, output_path):
    """Synthesize speech using VoiceDesign server."""
    print(f"🌐 Server: {remote_url}", file=sys.stderr)
    print(f"🎙️  Voice: {voice_desc[:60]}...", file=sys.stderr)
    
    try:
        response = requests.post(
            f"{remote_url}/tts",
            json={
                "text": text,
                "language": language,
                "voice_description": voice_desc,
                "instruct": instruct
            },
            timeout=180
        )
        
        if response.status_code != 200:
            print(f"❌ Server error: {response.status_code}", file=sys.stderr)
            print(response.text, file=sys.stderr)
            sys.exit(1)
        
        # Save audio
        output_path = Path(output_path).expanduser().resolve()
        output_path.parent.mkdir(parents=True, exist_ok=True)
        
        with open(output_path, 'wb') as f:
            f.write(response.content)
        
        print(f"✅ Audio saved: {output_path}", file=sys.stderr)
        print(str(output_path))
        
    except requests.exceptions.RequestException as e:
        print(f"❌ Connection error: {e}", file=sys.stderr)
        print(f"   Is server running at {remote_url}?", file=sys.stderr)
        sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description="Qwen3-TTS VoiceDesign Client",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog=__doc__
    )
    
    parser.add_argument("text", help="Text to synthesize")
    parser.add_argument("-o", "--output", default="qwen_output.wav",
                       help="Output audio file path")
    parser.add_argument("-v", "--voice", default=DEFAULT_VOICE,
                       help="Voice description")
    parser.add_argument("-l", "--language", default="Italian",
                       help="Language (default: Italian)")
    parser.add_argument("-i", "--instruct", default="",
                       help="Additional instructions")
    parser.add_argument("--remote",
                       default=os.environ.get("QWEN_TTS_REMOTE"),
                       help="Remote server URL (required)")
    
    args = parser.parse_args()
    
    if not args.remote:
        print("❌ Error: --remote URL required", file=sys.stderr)
        print("   Or set: export QWEN_TTS_REMOTE=http://192.168.188.177:8765", file=sys.stderr)
        sys.exit(1)
    
    synthesize(
        remote_url=args.remote,
        text=args.text,
        voice_desc=args.voice,
        language=args.language,
        instruct=args.instruct,
        output_path=args.output
    )


if __name__ == "__main__":
    main()
