#!/usr/bin/env bash
# Setup script for Qwen3-TTS skill
# Run this manually to create the virtual environment and install dependencies

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
VENV_DIR="$SKILL_DIR/venv"

echo "🔧 Qwen3-TTS Setup"
echo "=================="
echo

# Check Python version
PYTHON_CMD=""
for cmd in python3.12 python3.11 python3.10 python3; do
    if command -v "$cmd" &> /dev/null; then
        VERSION=$($cmd --version 2>&1 | grep -oP '\d+\.\d+')
        MAJOR=$(echo "$VERSION" | cut -d. -f1)
        MINOR=$(echo "$VERSION" | cut -d. -f2)
        
        if [ "$MAJOR" -eq 3 ] && [ "$MINOR" -ge 10 ] && [ "$MINOR" -le 12 ]; then
            PYTHON_CMD="$cmd"
            echo "✓ Found suitable Python: $cmd ($VERSION)"
            break
        fi
    fi
done

if [ -z "$PYTHON_CMD" ]; then
    echo "❌ Error: Python 3.10-3.12 required (onnxruntime compatibility)"
    echo "   Current python3: $(python3 --version 2>&1)"
    exit 1
fi

# Create virtual environment
if [ -d "$VENV_DIR" ]; then
    echo "⚠️  Virtual environment already exists at: $VENV_DIR"
    read -p "Remove and recreate? (y/N): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        rm -rf "$VENV_DIR"
    else
        echo "Keeping existing venv. Run with --force to override."
        exit 0
    fi
fi

echo "📦 Creating virtual environment..."
$PYTHON_CMD -m venv "$VENV_DIR"

# Activate and install
echo "📥 Installing dependencies..."
source "$VENV_DIR/bin/activate"

# Upgrade pip first
pip install --upgrade pip setuptools wheel

# Install qwen-tts and dependencies
echo "   Installing qwen-tts (this may take several minutes)..."
pip install qwen-tts soundfile

echo
echo "✅ Setup complete!"
echo
echo "Virtual environment created at:"
echo "  $VENV_DIR"
echo
echo "Test with:"
echo "  scripts/qwen_tts.py \"Ciao, questo è un test\" -o test.wav"
echo
