#!/usr/bin/env python3
"""
Qwen-TTS HTTP Server

Run this on your Mac to serve TTS via HTTP API.

Installation:
  pip install qwen-tts soundfile fastapi uvicorn

Usage:
  python3 server.py --host 0.0.0.0 --port 8765

API:
  POST /tts
  Body: {
    "text": "Text to speak",
    "language": "Italian",
    "speaker": "Vivian",
    "instruct": "Optional instruction"
  }
  Returns: WAV audio file
"""

import argparse
import io
import sys
from pathlib import Path

try:
    import torch
    import soundfile as sf
    from qwen_tts import Qwen3TTSModel
    from fastapi import FastAPI, HTTPException
    from fastapi.responses import StreamingResponse
    from pydantic import BaseModel
    import uvicorn
except ImportError as e:
    print(f"Error: Missing dependency: {e}", file=sys.stderr)
    print("\nInstall with:", file=sys.stderr)
    print("  pip install qwen-tts soundfile fastapi uvicorn", file=sys.stderr)
    sys.exit(1)


# Request model
class TTSRequest(BaseModel):
    text: str
    language: str = "Auto"
    speaker: str = "Vivian"
    instruct: str = ""
    model: str = "Qwen/Qwen3-TTS-12Hz-0.6B-CustomVoice"


# Global model (loaded once)
tts_model = None
current_model_name = None


def load_model(model_name: str):
    """Load TTS model (cached globally)."""
    global tts_model, current_model_name
    
    if tts_model is not None and current_model_name == model_name:
        return tts_model
    
    print(f"Loading model: {model_name}...", file=sys.stderr)
    
    # Determine device
    if torch.cuda.is_available():
        device = "cuda:0"
        dtype = torch.bfloat16
        attn_impl = "flash_attention_2"
    elif torch.backends.mps.is_available():
        device = "mps"
        dtype = torch.float32  # Use float32 on MPS (float16 has numerical issues)
        attn_impl = "eager"
    else:
        device = "cpu"
        dtype = torch.float32
        attn_impl = "eager"
    
    try:
        tts_model = Qwen3TTSModel.from_pretrained(
            model_name,
            device_map=device,
            dtype=dtype,
            attn_implementation=attn_impl,
        )
        current_model_name = model_name
        print(f"Model loaded on {device}", file=sys.stderr)
        return tts_model
    except Exception as e:
        print(f"Error loading model: {e}", file=sys.stderr)
        raise


# FastAPI app
app = FastAPI(title="Qwen-TTS Server", version="1.0")


@app.get("/")
def root():
    """Health check."""
    return {
        "status": "ok",
        "model": current_model_name or "not loaded",
        "device": "cuda" if torch.cuda.is_available() else "mps" if torch.backends.mps.is_available() else "cpu"
    }


@app.get("/speakers")
def list_speakers():
    """List available speakers."""
    speakers = {
        "Vivian": {"lang": "Chinese", "desc": "Bright, slightly edgy young female"},
        "Serena": {"lang": "Chinese", "desc": "Warm, gentle young female"},
        "Uncle_Fu": {"lang": "Chinese", "desc": "Seasoned male, low mellow timbre"},
        "Dylan": {"lang": "Chinese (Beijing)", "desc": "Youthful Beijing male, clear"},
        "Eric": {"lang": "Chinese (Sichuan)", "desc": "Lively Chengdu male, husky"},
        "Ryan": {"lang": "English", "desc": "Dynamic male, rhythmic"},
        "Aiden": {"lang": "English", "desc": "Sunny American male"},
        "Ono_Anna": {"lang": "Japanese", "desc": "Playful female, light nimble"},
        "Sohee": {"lang": "Korean", "desc": "Warm female, rich emotion"},
    }
    return {"speakers": speakers}


@app.post("/tts")
def synthesize(request: TTSRequest):
    """
    Generate speech from text.
    
    Returns WAV audio file.
    """
    try:
        # Load model
        model = load_model(request.model)
        
        # Generate speech
        print(f"Synthesizing: {request.text[:60]}...", file=sys.stderr)
        wavs, sr = model.generate_custom_voice(
            text=request.text,
            language=request.language,
            speaker=request.speaker,
            instruct=request.instruct if request.instruct else None,
        )
        
        # Convert to WAV bytes
        buffer = io.BytesIO()
        sf.write(buffer, wavs[0], sr, format='WAV')
        buffer.seek(0)
        
        print(f"Generated {len(buffer.getvalue())} bytes", file=sys.stderr)
        
        return StreamingResponse(
            buffer,
            media_type="audio/wav",
            headers={
                "Content-Disposition": "attachment; filename=speech.wav"
            }
        )
        
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        raise HTTPException(status_code=500, detail=str(e))


def main():
    parser = argparse.ArgumentParser(description="Qwen-TTS HTTP Server")
    parser.add_argument("--host", default="0.0.0.0", help="Host to bind")
    parser.add_argument("--port", type=int, default=8765, help="Port to bind")
    parser.add_argument("--model", default="Qwen/Qwen3-TTS-12Hz-0.6B-CustomVoice",
                       help="Model to preload")
    parser.add_argument("--preload", action="store_true",
                       help="Preload model on startup")
    
    args = parser.parse_args()
    
    # Preload model if requested
    if args.preload:
        print(f"Preloading model: {args.model}")
        load_model(args.model)
    
    print(f"Starting Qwen-TTS server on {args.host}:{args.port}")
    print(f"API: http://{args.host}:{args.port}/docs")
    
    uvicorn.run(app, host=args.host, port=args.port)


if __name__ == "__main__":
    main()
